<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Services\ModuleUploader;
use App\Services\ModuleValidator;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\File;
use Nwidart\Modules\Facades\Module;

class ModuleManagerController extends Controller
{
    protected $moduleValidator;

    protected $moduleUploader;

    public function __construct(ModuleValidator $moduleValidator, ModuleUploader $moduleUploader)
    {
        $this->moduleValidator = $moduleValidator;
        $this->moduleUploader  = $moduleUploader;
    }

    /**
     * Display the module management page
     */
    public function index()
    {
        // In ModuleManagerController@index
        $modules = collect(Module::all())->map(function ($module) {
            return [
                'name'       => $module->getName(),
                'is_enabled' => $module->isEnabled(), // Boolean property, not a method
                'is_core'    => $this->moduleValidator->isCoreModule($module->getName()),
                'json'       => $this->getModuleJson($module),
            ];
        })->values();

        $page    = request()->get('page', 1);
        $perPage = 10;

        $items = $modules->slice(($page - 1) * $perPage, $perPage);

        $paginator = new \Illuminate\Pagination\LengthAwarePaginator(
            $items,
            $modules->count(),
            $perPage,
            $page,
            ['path' => request()->url()]
        );

        return view('admin.modules.index', ['modules' => $paginator]);
    }

    /**
     * Get module.json contents
     */
    protected function getModuleJson($module)
    {
        $path = base_path('Modules') . '/' . $module->getName() . '/module.json';

        return File::exists($path) ? json_decode(File::get($path), true) : [];
    }

    /**
     * Handle module upload
     */
    public function upload(Request $request)
    {
        $request->validate([
            'module' => 'required|file|mimes:zip',
        ]);

        $result = $this->moduleUploader->upload($request->file('module'));

        if ($result['status']) {
            return redirect()->back()->with('success', $result['message']);
        } else {
            return redirect()->back()->with('error', $result['message']);
        }
    }

    /**
     * Install a module
     */
    public function install($moduleName)
    {
        try {
            if (! Module::has($moduleName)) {
                return redirect()->back()->with('error', 'Module not found');
            }

            $module = Module::find($moduleName);

            // Run migrations
            Artisan::call('module:migrate', ['module' => $moduleName]);

            // Enable the module
            $module->enable();

            // Run seeders if they exist
            if (File::exists(base_path('Modules') . '/' . $moduleName . '/Database/Seeders')) {
                Artisan::call('module:seed', ['module' => $moduleName]);
            }

            // Publish module assets
            $this->publishModuleAssets($moduleName);

            // Clear caches
            Artisan::call('config:clear');
            Artisan::call('cache:clear');
            Artisan::call('route:clear');

            return redirect()->back()->with('success', "Module '{$moduleName}' installed successfully");
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Installation failed: ' . $e->getMessage());
        }
    }

    /**
     * Publish module assets
     */
    protected function publishModuleAssets(string $moduleName): void
    {
        $publishers = [
            'config'       => 'module:publish-config',
            'migrations'   => 'module:publish-migration',
            'assets'       => 'module:publish-asset',
            'views'        => 'module:publish-views',
            'translations' => 'module:publish-translation',
        ];

        foreach ($publishers as $type => $command) {
            try {
                Artisan::call($command, ['module' => $moduleName]);
            } catch (\Exception $e) {
                // Continue with other publishers even if one fails
            }
        }
    }

    /**
     * Toggle module enabled/disabled state
     */
    public function toggle($moduleName)
    {
        try {
            if ($this->moduleValidator->isCoreModule($moduleName)) {
                return redirect()->back()->with('error', 'Core modules cannot be disabled');
            }

            $module = Module::find($moduleName);

            if (! $module) {
                return redirect()->back()->with('error', 'Module not found');
            }

            if ($module->isEnabled()) {
                $module->disable();
                $status = 'disabled';
            } else {
                $module->enable();
                $status = 'enabled';
            }

            // Clear caches
            Artisan::call('config:clear');
            Artisan::call('cache:clear');

            return redirect()->back()->with('success', "Module '{$moduleName}' {$status} successfully");
        } catch (\Exception $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    /**
     * Delete a module
     */
    public function delete($moduleName)
    {
        try {
            // Prevent deleting core modules
            if ($this->moduleValidator->isCoreModule($moduleName)) {
                return redirect()->back()->with('error', 'Core modules cannot be deleted');
            }

            $module = Module::find($moduleName);
            if (! $module) {
                return redirect()->back()->with('error', 'Module not found');
            }

            // Disable module first
            if ($module->isEnabled()) {
                $module->disable();

                // Rollback migrations
                try {
                    Artisan::call('module:migrate-rollback', ['module' => $moduleName]);
                } catch (\Exception $e) {
                    // Continue even if migrations fail
                }
            }

            // Remove module from modules.json
            $modulesJson = json_decode(File::get(base_path('modules_statuses.json')), true);
            if (isset($modulesJson[$moduleName])) {
                unset($modulesJson[$moduleName]);
                File::put(base_path('modules_statuses.json'), json_encode($modulesJson, JSON_PRETTY_PRINT));
            }

            // Create backup before deletion
            $modulePath = base_path('Modules') . '/' . $moduleName;
            $backupPath = storage_path('app/module_backups') . '/' . $moduleName . '_' . date('Y-m-d-His');

            if (! File::exists(dirname($backupPath))) {
                File::makeDirectory(dirname($backupPath), 0755, true);
            }

            if (File::exists($modulePath)) {
                File::copyDirectory($modulePath, $backupPath);
                File::deleteDirectory($modulePath);
            }

            // Clear caches
            Artisan::call('config:clear');
            Artisan::call('cache:clear');

            return redirect()->back()->with('success', "Module '{$moduleName}' deleted successfully (a backup was created)");
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Deletion failed: ' . $e->getMessage());
        }
    }
}
