<?php

namespace App\Http\Controllers\WhatsApp;

use App\Http\Controllers\Controller;
use App\Models\BotFlow;
use App\Models\CampaignDetail;
use App\Models\Chat;
use App\Models\ChatMessage;
use App\Models\Contact;
use App\Models\MessageBots;
use App\Models\TemplateBot;
use App\Services\PusherService;
use App\Traits\WhatsApp;
use Illuminate\Http\Request;
use Netflie\WhatsAppCloudApi\Message\Media\LinkID;
use Netflie\WhatsAppCloudApi\WhatsAppCloudApi;

class WhatsAppWebhookController extends Controller
{
    use WhatsApp;

    public $is_first_time = false;

    public $is_bot_stop = false;

    /**
     * Handle incoming WhatsApp webhook requests
     *
     * @return \Illuminate\Http\Response
     */
    public function __invoke(Request $request)
    {
        whatsapp_log(
            'Webhook Received',
            'debug',
            [
                'method'           => $request->method(),
                'full_url'         => $request->fullUrl(),
                'hub_mode'         => $request->input('hub_mode'),
                'hub_verify_token' => $request->input('hub_verify_token'),
            ]
        );

        // WhatsApp Webhook Verification
        if (isset($_GET['hub_mode']) && isset($_GET['hub_challenge']) && isset($_GET['hub_verify_token'])) {
            // Retrieve verify token from settings
            $verifyToken = get_setting('whatsapp.webhook_verify_token');

            whatsapp_log(
                'Webhook Verification Attempt',
                'debug',
                [
                    'received_token' => $_GET['hub_verify_token'],
                    'expected_token' => $verifyToken,
                    'hub_mode'       => $_GET['hub_mode'],
                    'hub_challenge'  => $_GET['hub_challenge'],
                ]
            );

            // Verify the webhook
            if ($_GET['hub_verify_token'] == $verifyToken && $_GET['hub_mode'] == 'subscribe') {
                whatsapp_log(
                    'Webhook Verification Successful',
                    'info'
                );

                // Directly output the challenge with proper headers
                header('Content-Type: text/plain');
                echo $_GET['hub_challenge'];
                exit;
            } else {
                whatsapp_log(
                    'Webhook Verification Failed',
                    'warning',
                    [
                        'received_token' => $_GET['hub_verify_token'],
                        'hub_mode'       => $_GET['hub_mode'],
                    ]
                );

                // Send 403 Forbidden with a clear error message
                header('HTTP/1.1 403 Forbidden');
                header('Content-Type: text/plain');
                echo 'Verification failed: Invalid token or mode';
                exit;
            }
        }

        // Process webhook payload for messages and statuses
        $this->processWebhookPayload();
    }

    /**
     * Process incoming webhook payload
     */
    protected function processWebhookPayload()
    {
        $feedData = file_get_contents('php://input');

        whatsapp_log(
            'Webhook Payload Received',
            'info',
            [
                'payload' => $feedData,
            ]
        );

        if (! empty($feedData)) {

            $payload = json_decode($feedData, true);

            // Special ping message handling
            if (isset($payload['message']) && $payload['message'] === 'ctl_whatsbot_ping' && isset($payload['identifier'])) {
                whatsapp_log(
                    'Whatsmark Ping Received',
                    'debug'
                );
                echo json_encode(['status' => true, 'message' => 'Webhook verified']);

                return;
            }

            // Check for message ID to prevent duplicate processing
            $message_id = $payload['entry'][0]['changes'][0]['value']['messages'][0]['id'] ?? '';
            if (! empty($message_id)) {
                // Check if message already processed (similar to original code)
                $found = $this->checkMessageProcessed($message_id);
                if ($found) {
                    whatsapp_log(
                        'Duplicate Message Detected',
                        'warning',
                        [
                            'message_id' => $message_id,
                        ]
                    );

                    return;
                }
            }

            // Process the payload
            $this->processPayloadData($payload);

            // Forward webhook data if enabled
            $this->forwardWebhookData($feedData, $payload);
        }
    }

    /**
     * Check if message has already been processed
     */
    protected function checkMessageProcessed(string $messageId): bool
    {
        // Implement logic to check if message is already in database
        // Similar to original CI code: check in 'wtc_interaction_messages' table
        return \DB::table('chat_messages')
            ->where('message_id', $messageId)
            ->exists();
    }

    /**
     * Process payload data
     */
    protected function processPayloadData(array $payload)
    {
        whatsapp_log(
            'Processing Payload Data',
            'info',
            [
                'payload_entries' => count($payload['entry']),
            ]
        );

        // Extract entry and changes
        $entry   = array_shift($payload['entry']);
        $changes = array_shift($entry['changes']);
        $value   = $changes['value'];

        // Process messages or statuses
        if (isset($value['messages'])) {
            $this->processIncomingMessages($value);
            $this->processBotSending($value);
        } elseif (isset($value['statuses'])) {
            $this->processMessageStatuses($value['statuses']);
        }
    }

    private function processBotSending(array $message_data)
    {
        if (! empty($message_data['messages'])) {
            $message     = reset($message_data['messages']);
            $trigger_msg = isset($message['button']['text']) ? $message['button']['text'] : $message['text']['body'] ?? '';
            if (! empty($message['interactive']) && $message['interactive']['type'] == 'button_reply') {
                $trigger_msg = $message['interactive']['button_reply']['id'];
            }
            $contact  = reset($message_data['contacts']);
            $metadata = $message_data['metadata'];

            try {
                $contact_number = $message['from'];
                $contact_data   = $this->getContactData($contact_number, $contact['profile']['name']);

                $query_trigger_msg = $trigger_msg;
                $reply_type        = null;
                if ($this->is_first_time) {
                    $query_trigger_msg = '';
                    $reply_type        = 3;
                }

                $current_interaction = Chat::where(['type' => $contact_data->type, 'type_id' => $contact_data->id, 'wa_no' => $message_data['metadata']['display_phone_number']])->first();

                if ($current_interaction->is_bots_stoped == 1 && (time() > strtotime($current_interaction->bot_stoped_time) + ((int) get_setting('whats-mark.restart_bots_after') * 3600))) {
                    Chat::where('id', $current_interaction->id)->update(['bot_stoped_time' => null, 'is_bots_stoped' => '0']);
                    $this->is_bot_stop = false;
                } elseif ($current_interaction->is_bots_stoped == 1) {
                    $this->is_bot_stop = true;
                }

                if (collect(get_setting('whats-mark.stop_bots_keyword'))->first(fn ($keyword) => str_contains($trigger_msg, $keyword))) {
                    Chat::where('id', $current_interaction->id)->update(['bot_stoped_time' => date('Y-m-d H:i:s'), 'is_bots_stoped' => '1']);
                    $this->is_bot_stop = true;
                }

                if (! $this->is_bot_stop) {
                    // Fetch template and message bots based on interaction
                    $template_bots = TemplateBot::getTemplateBotsByRelType($contact_data->type ?? '', $query_trigger_msg, $reply_type);
                    $message_bots  = MessageBots::getMessageBotsbyRelType($contact_data->type ?? '', $query_trigger_msg, $reply_type);

                    if (empty($template_bots) && empty($message_bots)) {
                        $template_bots = TemplateBot::getTemplateBotsByRelType($contact_data->type ?? '', $query_trigger_msg, 4);
                        $message_bots  = MessageBots::getMessageBotsbyRelType($contact_data->type ?? '', $query_trigger_msg, 4);
                    }

                    $add_messages = function ($item) {
                        $item['header_message'] = $item['header_data_text'];
                        $item['body_message']   = $item['body_data'];
                        $item['footer_message'] = $item['footer_data'];

                        return $item;
                    };

                    $template_bots = array_map($add_messages, $template_bots);

                    // Iterate over template bots
                    foreach ($template_bots as $template) {
                        $template['rel_id'] = $contact_data->id;
                        if (! empty($contact_data->userid)) {
                            $template['userid'] = $contact_data->userid;
                        }

                        // Send template on exact match, contains, or first time
                        if ((1 == $template['reply_type'] && in_array(strtolower($trigger_msg), array_map('trim', array_map('strtolower', explode(',', $template['trigger']))))) || (2 == $template['reply_type'] && ! empty(array_filter(explode(',', strtolower($template['trigger'])), fn ($word) => preg_match('/\b' . preg_quote(trim($word), '/') . '\b/', strtolower($trigger_msg))))) || (3 == $template['reply_type'] && $this->is_first_time) || 4 == $template['reply_type']) {
                            $response    = $this->sendTemplate($contact_number, $template, 'template_bot', $metadata['phone_number_id']);
                            $chatId      = $this->createOrUpdateInteraction($contact_number, $message_data['metadata']['display_phone_number'], $message_data['metadata']['phone_number_id'], $contact_data->firstname . ' ' . $contact_data->lastname, '', '', false);
                            $chatMessage = $this->storeBotMessages($template, $chatId, $contact_data, 'template_bot', $response);
                        }
                    }

                    // Iterate over message bots
                    foreach ($message_bots as $message) {
                        $message['rel_id'] = $contact_data->id;
                        if (! empty($contact_data->userid)) {
                            $message['userid'] = $contact_data->userid;
                        }
                        if ((1 == $message['reply_type'] && in_array(strtolower($trigger_msg), array_map('trim', array_map('strtolower', explode(',', $message['trigger']))))) || (2 == $message['reply_type'] && ! empty(array_filter(explode(',', strtolower($message['trigger'])), fn ($word) => preg_match('/\b' . preg_quote(trim($word), '/') . '\b/', strtolower($trigger_msg))))) || (3 == $message['reply_type'] && $this->is_first_time) || 4 == $message['reply_type']) {
                            $response    = $this->sendMessage($contact_number, $message, $metadata['phone_number_id']);
                            $chatId      = $this->createOrUpdateInteraction($contact_number, $message_data['metadata']['display_phone_number'], $message_data['metadata']['phone_number_id'], $contact_data->firstname . ' ' . $contact_data->lastname, '', '', false);
                            $chatMessage = $this->storeBotMessages($message, $chatId, $contact_data, '', $response);
                        }
                    }
                }
            } catch (\Throwable $th) {
                file_put_contents(base_path() . '/errors.json', json_encode([$th->getMessage()]));
            }
        }
        $this->processBotFlow($message_data);
    }

    /**
     * Process incoming messages
     */
    protected function processIncomingMessages(array $value)
    {
        $messageEntry   = array_shift($value['messages']);
        $contact        = array_shift($value['contacts']) ?? '';
        $name           = $contact['profile']['name']     ?? '';
        $from           = $messageEntry['from'];
        $metadata       = $value['metadata'];
        $wa_no          = $metadata['display_phone_number'];
        $wa_no_id       = $metadata['phone_number_id'];
        $messageType    = $messageEntry['type'];
        $message_id     = $messageEntry['id'];
        $ref_message_id = isset($messageEntry['context']) ? $messageEntry['context']['id'] : '';

        // Determine if this is a first-time interaction
        $this->is_first_time = $this->isFirstTimeInteraction($from);

        // Extract message content based on type
        $message = $this->extractMessageContent($messageEntry, $messageType);
        if ($messageType == 'image' || $messageType == 'audio' || $messageType == 'document' || $messageType == 'video') {
            $media_id   = $messageEntry[$messageType]['id'];
            $attachment = $this->retrieveUrl($media_id);
        }

        whatsapp_log(
            'Processing Incoming Message',
            'info',
            [
                'from'          => $from,
                'name'          => $name,
                'message_type'  => $messageType,
                'is_first_time' => $this->is_first_time,
            ]
        );

        // Create or update interaction
        $interaction_id = $this->createOrUpdateInteraction(
            $from,
            $wa_no,
            $wa_no_id,
            $name,
            $message,
            $messageType
        );

        // Store interaction message
        $message_id = $this->storeInteractionMessage(
            $interaction_id,
            $from,
            $message_id,
            $message,
            $messageType,
            $ref_message_id,
            $metadata,
            $attachment ?? ''
        );

        if (! empty(get_setting('pusher.app_key')) && ! empty(get_setting('pusher.app_secret')) && ! empty(get_setting('pusher.app_id')) && ! empty(get_setting('pusher.cluster'))) {
            $pusherService = new PusherService;
            $pusherService->trigger('whatsmark-chat-channel', 'whatsmark-chat-event', [
                'chat' => ChatController::newChatMessage($interaction_id, $message_id),
            ]);
        }
    }

    /**
     * Check if this is a first-time interaction
     */
    protected function isFirstTimeInteraction(string $from): bool
    {
        return ! (bool) Chat::where('receiver_id', $from)->count();
    }

    /**
     * Extract message content based on type
     */
    private function extractFlowMessageContent($nodeData, $nodeType)
    {
        $output = $nodeData['output'][0] ?? [];

        switch ($nodeType) {
            case 'textMessage':
                return $output['reply_text'] ?? '';
            case 'buttonMessage':
                return $output['reply_text'] ?? '';
            case 'listMessage':
                return $output['reply_text'] ?? '';
            case 'callToAction':
                return $output['reply_text'] ?? $output['bot_header'] ?? '';
            case 'mediaMessage':
                return $output['media_caption'] ?? 'Media message';
            case 'locationMessage':
                return $output['location_name'] ?? 'Location message';
            case 'contactMessage':
                return 'Contact message';
            default:
                return json_encode($output);
        }
    }

    protected function extractMessageContent(array $messageEntry, string $messageType): string
    {
        switch ($messageType) {
            case 'text':
                return $messageEntry['text']['body'] ?? '';
            case 'interactive':
                return $messageEntry['interactive']['button_reply']['title'] ?? '';
            case 'button':
                return $messageEntry['button']['text'] ?? '';
            case 'reaction':
                return json_decode('"' . ($messageEntry['reaction']['emoji'] ?? '') . '"', false, 512, JSON_UNESCAPED_UNICODE);
            case 'image':
            case 'audio':
            case 'document':
            case 'video':
                return $messageType;
            case 'contacts':
                return json_encode($messageEntry['contacts']);
            case 'location':
                return json_encode($messageEntry['location']);
            default:
                return 'Unknown message type';
        }
    }

    /**
     * Create or update interaction
     */
    protected function createOrUpdateInteraction(
        string $from,
        string $wa_no,
        string $wa_no_id,
        string $name,
        string $message,
        string $messageType,
        bool $enableTime = true
    ): int {
        // Retrieve contact data (similar to original implementation)
        $contact_data = $this->getContactData($from, $name);

        // Check if a record with the same receiver_id exists
        $existingChat = Chat::where(['receiver_id' => $from, 'wa_no' => $wa_no])->first();

        if ($existingChat) {

            Chat::where('id', $existingChat->id)->update([
                'wa_no'         => $wa_no,
                'wa_no_id'      => $wa_no_id,
                'name'          => $name,
                'last_message'  => $message,
                'last_msg_time' => now(),
                'type'          => $contact_data->type ?? 'guest',
                'type_id'       => $contact_data->id   ?? '',
                'updated_at'    => now(),
            ] + ($enableTime ? ['time_sent' => now()] : []));

            return $existingChat->id;
        } else {

            return Chat::insertGetId([
                'receiver_id'   => $from,
                'wa_no'         => $wa_no,
                'wa_no_id'      => $wa_no_id,
                'name'          => $name,
                'last_message'  => $message,
                'agent'         => json_encode(['assign_id' => $contact_data->assigned_id ?? 0, 'agents_id' => '']),
                'time_sent'     => now(),
                'last_msg_time' => now(),
                'type'          => $contact_data->type ?? 'guest',
                'type_id'       => $contact_data->id   ?? '',
                'created_at'    => now(),
                'updated_at'    => now(),
            ]);
        }
    }

    /**
     * Store interaction message
     */
    protected function storeInteractionMessage(
        int $interaction_id,
        string $from,
        string $message_id,
        string $message,
        string $messageType,
        string $ref_message_id,
        array $metadata,
        string $url = ''
    ) {
        return ChatMessage::insertGetId([
            'interaction_id' => $interaction_id,
            'sender_id'      => $from,
            'message_id'     => $message_id,
            'message'        => $message,
            'type'           => $messageType,
            'staff_id'       => null,
            'status'         => 'sent',
            'time_sent'      => now(),
            'ref_message_id' => $ref_message_id,
            'created_at'     => now(),
            'updated_at'     => now(),
            'url'            => $url,
        ]);
    }

    /**
     * Process message statuses
     */
    protected function processMessageStatuses(array $statuses)
    {
        foreach ($statuses as $status) {
            $id           = $status['id'];
            $status_value = $status['status'];

            $status_message = null;
            $errors         = $status['errors'] ?? [];

            $error_data = array_column($errors, 'error_data');
            $details    = array_column($error_data, 'details');

            $status_message = reset($details) ?: null;

            // Update chat message
            CampaignDetail::where('whatsapp_id', $id)->update(['message_status' => $status_value, 'response_message' => $status_message]);
            $message = ChatMessage::where('message_id', $id)->first();

            if ($message) {
                $message->update([
                    'status'         => $status_value,
                    'status_message' => $status_message,
                    'updated_at'     => now(),
                ]);

                if (! empty(get_setting('pusher.app_key')) && ! empty(get_setting('pusher.app_secret')) && ! empty(get_setting('pusher.app_id')) && ! empty(get_setting('pusher.cluster'))) {
                    $pusherService = new PusherService;
                    $pusherService->trigger('whatsmark-chat-channel', 'whatsmark-chat-event', [
                        'chat' => ChatController::newChatMessage($message->interaction_id, $message->id),
                    ]);
                }
            }
        }
    }

    /**
     * Forward webhook data if enabled
     */
    protected function forwardWebhookData(string $feedData, array $payload)
    {
        if (get_setting('whats-mark.enable_webhook_resend') && filter_var(get_setting('whats-mark.whatsapp_data_resend_to'), FILTER_VALIDATE_URL)) {
            try {
                $forwardMethod = get_setting('whats-mark.webhook_resend_method', 'POST');
                $response      = \Http::send($forwardMethod, get_setting('whats-mark.whatsapp_data_resend_to'), [
                    'body' => $forwardMethod === 'POST' ? $feedData : $payload,
                ]);

                whatsapp_log(
                    'Webhook resend data',
                    'info',
                    [
                        'status' => $response->status(),
                        'data'   => $response->body(),
                    ]
                );
            } catch (\Exception $e) {

                whatsapp_log(
                    'Webhook Forward Error',
                    'error',
                    [
                        'message' => $e->getMessage(),
                    ],
                    $e
                );
            }
        }
    }

    /**
     * Get contact data (placeholder method)
     */
    protected function getContactData(string $from, string $name): object
    {
        $contact = Contact::whereRaw('phone = ? OR phone = ?', ['+' . $from, $from])->first();
        if ($contact) {
            return $contact;
        }
        if (get_setting('whats-mark.auto_lead_enabled')) {
            $name    = explode(' ', $name);
            $contact = Contact::create([
                'firstname'   => $name[0],
                'lastname'    => count($name) > 1 ? implode(' ', array_slice($name, 1)) : '',
                'type'        => 'lead',
                'phone'       => $from[0] === '+' ? $from : '+' . $from,
                'assigned_id' => get_setting('whats-mark.lead_assigned_to'),
                'status_id'   => get_setting('whats-mark.lead_status'),
                'source_id'   => get_setting('whats-mark.lead_source'),
                'addedfrom'   => '0',
            ]);

            return $contact;
        }

        return (object) [];
    }

    public function storeBotMessages($data, $interactionId, $relData, $type, $response)
    {
        $data['sending_count'] = (int) $data['sending_count'] + 1;

        if ($type == 'template_bot') {
            $header = parseText($data['rel_type'], 'header', $data);
            $body   = parseText($data['rel_type'], 'body', $data);
            $footer = parseText($data['rel_type'], 'footer', $data);

            $buttonHtml = '';
            if (! empty(json_decode($data['buttons_data']))) {
                $buttons    = json_decode($data['buttons_data']);
                $buttonHtml = "<div class='flex flex-col mt-2 space-y-2'>";
                foreach ($buttons as $button) {
                    $buttonHtml .= "<button class='bg-gray-100 text-success-500 px-3 py-2 rounded-lg flex items-center justify-center text-xs space-x-2 w-full
                        dark:bg-gray-800 dark:text-success-400'>" . e($button->text) . '</button>';
                }
                $buttonHtml .= '</div>';
            }

            $headerData     = '';
            $fileExtensions = get_meta_allowed_extension();
            $extension      = strtolower(pathinfo($data['filename'], PATHINFO_EXTENSION));
            $fileType       = array_key_first(array_filter($fileExtensions, fn ($data) => in_array('.' . $extension, explode(', ', $data['extension']))));
            if ($data['header_data_format'] === 'IMAGE' && $fileType == 'image') {
                $headerData = "<a href='" . asset('storage/templates/' . $data['filename']) . "' data-lightbox='image-group'>
                <img src='" . asset('storage/' . $data['filename']) . "' class='rounded-lg w-full mb-2'>
            </a>";
            } elseif ($data['header_data_format'] === 'TEXT' || $data['header_data_format'] === '') {
                $headerData = "<span class='font-bold mb-3'>" . nl2br(decodeWhatsAppSigns(e($header ?? ''))) . '</span>';
            } elseif ($data['header_data_format'] === 'DOCUMENT') {
                $headerData = "<a href='" . asset('storage/' . $data['filename']) . "' target='_blank' class='btn btn-secondary w-full'>" . t('document') . '</a>';
            } elseif ($data['header_data_format'] === 'VIDEO') {
                $headerData = "<video src='" . asset('storage/' . $data['filename']) . "' controls class='rounded-lg w-full'></video>";
            }

            TemplateBot::where('id', $data['id'])->update(['sending_count' => $data['sending_count'] + 1]);

            $chat_message = [
                'interaction_id' => $interactionId,
                'sender_id'      => get_setting('whatsapp.wm_default_phone_number'),
                'url'            => null,
                'message'        => "
                $headerData
                <p>" . nl2br(decodeWhatsAppSigns(e($body))) . "</p>
                <span class='text-gray-500 text-sm'>" . nl2br(decodeWhatsAppSigns(e($footer ?? ''))) . "</span>
                $buttonHtml
            ",
                'status'     => 'sent',
                'time_sent'  => now()->toDateTimeString(),
                'message_id' => $response['data']->messages[0]->id ?? null,
                'staff_id'   => 0,
                'type'       => 'text',
            ];

            $message_id = ChatMessage::insertGetId($chat_message);

            if (! empty(get_setting('pusher.app_key')) && ! empty(get_setting('pusher.app_secret')) && ! empty(get_setting('pusher.app_id')) && ! empty(get_setting('pusher.cluster'))) {
                $pusherService = new PusherService;
                $pusherService->trigger('whatsmark-chat-channel', 'whatsmark-chat-event', [
                    'chat' => ChatController::newChatMessage($interactionId, $message_id),
                ]);
            }

            return $message_id;
        }

        $type   = $type === 'flow' ? 'flow' : 'bot_files';
        $data   = parseMessageText($data);
        $header = $data['bot_header'] ?? '';
        $body   = $data['reply_text'] ?? '';
        $footer = $data['bot_footer'] ?? '';

        $headerImage       = '';
        $allowedExtensions = get_meta_allowed_extension();

        $buttonHtml = "<div class='flex flex-col mt-2 space-y-2'>";
        $option     = false;

        if (! empty($data['button1_id'])) {
            $buttonHtml .= "<button class='bg-gray-100 text-success-500 px-3 py-2 rounded-lg flex items-center justify-center text-xs space-x-2 w-full
               dark:bg-gray-800 dark:text-success-400'>" . e($data['button1']) . '</button>';
            $option = true;
        }
        if (! empty($data['button2_id'])) {
            $buttonHtml .= "<button class='bg-gray-100 text-success-500 px-3 py-2 rounded-lg flex items-center justify-center text-xs space-x-2 w-full
               dark:bg-gray-800 dark:text-success-400'>" . e($data['button2']) . '</button>';
            $option = true;
        }
        if (! empty($data['button3_id'])) {
            $buttonHtml .= "<button class='bg-gray-100 text-success-500 px-3 py-2 rounded-lg flex items-center justify-center text-xs space-x-2 w-full
               dark:bg-gray-800 dark:text-success-400'>" . e($data['button3']) . '</button>';
            $option = true;
        }
        if (! $option && ! empty($data['button_name']) && ! empty($data['button_url']) && filter_var($data['button_url'], FILTER_VALIDATE_URL)) {
            $buttonHtml .= "<a href='" . e($data['button_url']) . "' class='bg-gray-100 text-success-500 px-3 py-2 rounded-lg flex items-center justify-center text-xs space-x-2 w-full
               dark:bg-gray-800 dark:text-success-400 mt-2'> <svg class='w-4 h-4 text-success-500' aria-hidden='true' xmlns='http://www.w3.org/2000/svg' fill='none' viewBox='0 0 24 24'> <path stroke='currentColor' stroke-linecap='round' stroke-linejoin='round' stroke-width='2' d='M18 14v4.833A1.166 1.166 0 0 1 16.833 20H5.167A1.167 1.167 0 0 1 4 18.833V7.167A1.166 1.166 0 0 1 5.167 6h4.618m4.447-2H20v5.768m-7.889 2.121 7.778-7.778'/> </svg><span class='whitespace-nowrap'>" . e($data['button_name']) . '</a>';
            $option = true;
        }

        $extension = strtolower(pathinfo($data['filename'], PATHINFO_EXTENSION));
        $fileType  = array_key_first(array_filter($allowedExtensions, fn ($data) => in_array('.' . $extension, explode(', ', $data['extension']))));
        if (! $option && ! empty($data['filename']) && $fileType == 'image') {
            $headerImage = "<a href='" . asset('storage/' . $data['filename']) . "' data-lightbox='image-group'>
            <img src='" . asset('storage/' . $data['filename']) . "' class='rounded-lg w-full mb-2'>
        </a>";
        }
        if (! $option && ! empty($data['filename']) && $fileType == 'document') {
            $headerImage = "<a href='" . asset('storage/' . $data['filename']) . "' target='_blank' class='bg-gray-100 text-success-500 px-3 py-2 rounded-lg flex items-center justify-center text-xs space-x-2 w-full
               dark:bg-gray-800 dark:text-success-400'>" . t('document') . '</a>';
        }
        if (! $option && ! empty($data['filename']) && $fileType == 'video') {
            $headerImage = "<video src='" . asset('storage/' . $data['filename']) . "' controls class='rounded-lg w-full'></video>";
        }
        if (! $option && ! empty($data['filename']) && $fileType == 'audio') {
            $headerImage = "<audio controls class='w-64'><source src='" . asset('storage/' . $data['filename']) . "' type='audio/mpeg'></audio>";
        }
        $buttonHtml .= '</div>';

        MessageBots::where('id', $data['id'])->update(['sending_count' => $data['sending_count'] + 1]);

        $buttondata = $buttonHtml == "<div class='flex flex-col mt-2 space-y-2'></div>" ? '' : $buttonHtml;

        $chat_message = [
            'interaction_id' => $interactionId,
            'sender_id'      => get_setting('whatsapp.wm_default_phone_number'),
            'url'            => null,
            'message'        => $headerImage . "
            <span class='font-bold mb-3'>" . nl2br(e($header ?? '')) . '</span>
            <p>' . nl2br(decodeWhatsAppSigns(e($body))) . "</p>
            <span class='text-gray-500 text-sm'>" . nl2br(e($footer ?? '')) . "</span>
            $buttondata
        ",
            'status'     => 'sent',
            'time_sent'  => now()->toDateTimeString(),
            'message_id' => $response['data']->messages[0]->id ?? null,
            'staff_id'   => 0,
            'type'       => 'text',
        ];

        $message_id = ChatMessage::insertGetId($chat_message);

        if (! empty(get_setting('pusher.app_key')) && ! empty(get_setting('pusher.app_secret')) && ! empty(get_setting('pusher.app_id')) && ! empty(get_setting('pusher.cluster'))) {
            $pusherService = new PusherService;
            $pusherService->trigger('whatsmark-chat-channel', 'whatsmark-chat-event', [
                'chat' => ChatController::newChatMessage($interactionId, $message_id),
            ]);
        }

        return $message_id;
    }

    /**
     * Send a message via WhatsApp
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function send_message(Request $request)
    {
        try {
            // Get request data
            $id      = $request->input('id', '');
            $type    = $request->input('type');
            $type_id = $request->input('type_id');

            // Find existing chat/interaction
            $query = Chat::query();
            if (! empty($type_id)) {
                $query->where('type', $type)
                    ->where('type_id', $type_id);
            }
            $existing_interaction = $query->where('id', $id)->first();

            if (! $existing_interaction) {
                return response()->json(['error' => 'Interaction not found'], 404);
            }

            $to      = $existing_interaction->receiver_id;
            $message = strip_tags($request->input('message', ''));

            // Parse message text for contacts or leads
            $user_id = null;
            if ($type == 'customer' || $type == 'lead') {
                // Get user ID from contact (implement based on your database structure)
                $contact = Contact::find($type_id);
                $user_id = $contact->user_id ?? null;
            }

            $message_data = parseMessageText([
                'rel_type'   => $type,
                'rel_id'     => $type_id,
                'reply_text' => $message,
                'userid'     => $user_id,
            ]);

            $message = $message_data['reply_text'] ?? $message;

            $ref_message_id = $request->input('ref_message_id');

            $message_data = [];

            // Add text message if provided
            if (! empty($message)) {
                $message_data[] = [
                    'type' => 'text',
                    'text' => [
                        'preview_url' => true,
                        'body'        => $message,
                    ],
                ];
            }

            // Handle file attachments
            $attachments = [
                'audio'    => $request->file('audio'),
                'image'    => $request->file('image'),
                'video'    => $request->file('video'),
                'document' => $request->file('document'),
            ];

            foreach ($attachments as $type => $file) {
                if (! empty($file)) {
                    $file_url = $this->handle_attachment_upload($file);

                    $message_data[] = [
                        'type' => $type,
                        $type  => [
                            'url' => url('storage/whatsapp-attachments/' . $file_url),
                        ],
                    ];
                }
            }

            // Initialize WhatsApp Cloud API client
            $whatsapp_cloud_api = new WhatsAppCloudApi([
                'from_phone_number_id' => $existing_interaction->wa_no_id,
                'access_token'         => get_setting('whatsapp.wm_access_token'),
            ]);

            $messageId = null;

            // Send each message component
            foreach ($message_data as $data) {
                try {
                    switch ($data['type']) {
                        case 'text':
                            $response = $whatsapp_cloud_api->sendTextMessage($to, $data['text']['body']);
                            break;
                        case 'audio':
                            $response = $whatsapp_cloud_api->sendAudio($to, new LinkID($data['audio']['url']));
                            break;
                        case 'image':
                            $response = $whatsapp_cloud_api->sendImage($to, new LinkID($data['image']['url']));
                            break;
                        case 'video':
                            $response = $whatsapp_cloud_api->sendVideo($to, new LinkID($data['video']['url']));
                            break;
                        case 'document':
                            $fileName = basename($data['document']['url']);
                            $response = $whatsapp_cloud_api->sendDocument($to, new LinkID($data['document']['url']), $fileName, '');
                            break;
                        default:
                            continue 2;
                    }

                    // Decode the response JSON
                    $response_data = $response->decodedBody();

                    // Store the message ID if available
                    if (isset($response_data['messages'][0]['id'])) {
                        $messageId = $response_data['messages'][0]['id'];
                    }
                } catch (\Exception $e) {
                    whatsapp_log('Failed to send WhatsApp message', 'error', ['error' => $e->getMessage(), 'data' => $e], $e);

                    return response()->json(['success' => false, 'message' => 'Failed to send message: ' . $e->getMessage()]);
                }
            }

            // Create or update chat entry
            $interaction_id = $this->createOrUpdateInteraction($to, $existing_interaction->wa_no, $existing_interaction->wa_no_id, $existing_interaction->name, $message ?? ($message_data[0]['type'] ?? ''), '', false);

            foreach ($message_data as $data) {
                $message_id = ChatMessage::insertGetId([
                    'interaction_id' => $interaction_id,
                    'sender_id'      => $existing_interaction->wa_no,
                    'message'        => $message,
                    'message_id'     => $messageId,
                    'type'           => $data['type'] ?? '',
                    'staff_id'       => auth()->id(),
                    'url'            => isset($data[$data['type']]['url']) ? basename($data[$data['type']]['url']) : null,
                    'status'         => 'sent',
                    'time_sent'      => now(),
                    'ref_message_id' => $ref_message_id ?? '',
                    'created_at'     => now(),
                    'updated_at'     => now(),
                    'is_read'        => 1,
                ]);

                // Broadcast message via Pusher if enabled
                if (! empty(get_setting('pusher.app_key')) && ! empty(get_setting('pusher.app_secret')) && ! empty(get_setting('pusher.app_id')) && ! empty(get_setting('pusher.cluster'))) {
                    $pusherService = new PusherService;
                    $pusherService->trigger('whatsmark-chat-channel', 'whatsmark-chat-event', [
                        'chat' => ChatController::newChatMessage($interaction_id, $message_id),
                    ]);
                }
            }

            return response()->json(['success' => true]);
        } catch (\Exception $e) {
            whatsapp_log(
                'Failed to send WhatsApp message',
                'error',
                [
                    'error' => $e->getMessage(),
                    'data'  => $e,
                ],
                $e
            );

            return response()->json(['success' => false, 'message' => 'Failed to send message: ' . $e->getMessage()]);
        }
    }

    /**
     * Handle file attachment uploads
     *
     * @param  \Illuminate\Http\UploadedFile $file
     * @return string                        The stored file name
     */
    protected function handle_attachment_upload($file)
    {
        if (empty($file)) {
            return null;
        }

        $fileName = time() . '_' . $file->getClientOriginalName();
        $file->storeAs('whatsapp-attachments', $fileName, 'public');

        return $fileName;
    }

    /**
     * Process bot flow execution for incoming messages
     *
     * @param  array $message_data Message data from webhook
     * @return void
     */
    private function processBotFlow(array $message_data)
    {
        if (empty($message_data['messages'])) {
            return;
        }

        $message        = reset($message_data['messages']);
        $trigger_msg    = $this->extractTriggerMessage($message);
        $ref_message_id = isset($message['context']) ? $message['context']['id'] : null;

        // Use the new extraction method for both buttons and lists
        $button_id = $this->extractButtonIdFromMessage($message);

        whatsapp_log('processBotFlow - Extracted data', 'info', [
            'trigger_msg'      => $trigger_msg,
            'ref_message_id'   => $ref_message_id,
            'button_id'        => $button_id,
            'message_type'     => $message['type']                ?? 'unknown',
            'interactive_type' => $message['interactive']['type'] ?? 'none',
        ]);

        if (empty($trigger_msg) && empty($ref_message_id) && empty($button_id)) {
            whatsapp_log('No trigger, ref_message_id, or button_id found - exiting', 'info');

            return;
        }

        // ... rest of your processBotFlow method remains the same
        try {
            $contact_number = $message['from'];
            $contact        = reset($message_data['contacts']) ?? [];
            $metadata       = $message_data['metadata'];
            $contact_data   = $this->getContactData($contact_number, $contact['profile']['name'] ?? '');

            // Check if this is a first-time interaction
            $this->is_first_time = $this->isFirstTimeInteraction($contact_number);

            // Get current interaction/chat
            $current_interaction = Chat::where([
                'receiver_id' => $contact_number,
                'wa_no'       => $metadata['display_phone_number'],
            ])->first();

            if (! $current_interaction) {
                $interaction_id = $this->createOrUpdateInteraction(
                    $contact_number,
                    $metadata['display_phone_number'],
                    $metadata['phone_number_id'],
                    $contact['profile']['name'] ?? 'Guest',
                    $trigger_msg,
                    ''
                );
                $current_interaction = Chat::find($interaction_id);
            }

            // Check if bot is stopped
            if ($this->shouldSkipBotFlow($current_interaction, $trigger_msg)) {
                $this->is_bot_stop = true;

                return;
            }

            $this->is_bot_stop = false;

            // Find the right flow to execute
            $flow_execution = $this->determineFlowExecution(
                $contact_data,
                $trigger_msg,
                $button_id,
                $ref_message_id,
                $current_interaction->id,
                $contact_number,
                $metadata['phone_number_id']
            );

            whatsapp_log('Flow execution result', 'info', [
                'flow_execution_result' => $flow_execution,
                'button_id'             => $button_id,
                'ref_message_id'        => $ref_message_id,
            ]);

            if (! $flow_execution) {
                whatsapp_log('No flow execution - trying legacy bots', 'info');
                $this->processLegacyBots($contact_data, $trigger_msg, $current_interaction, $message_data);
            }
        } catch (\Throwable $th) {
            whatsapp_log('Error processing bot flow', 'error', [
                'error' => $th->getMessage(),
                'stack' => $th->getTraceAsString(),
            ], $th);
        }
    }

    /**
     * Determine which flow to execute based on incoming message and context
     * This is a complete replacement that focuses on processing ALL connected nodes
     */
    private function determineFlowExecution($contactData, $triggerMsg, $buttonId, $refMessageId, $chatId, $contactNumber, $phoneNumberId)
    {
        whatsapp_log('Determine flow execution (database-free)', 'info', [
            'trigger_msg'        => $triggerMsg,
            'button_id'          => $buttonId,
            'ref_message_id'     => $refMessageId,
            'is_button_response' => ! empty($buttonId),
        ]);

        // Handle button responses
        if ($buttonId) {
            whatsapp_log('Processing button response', 'info', [
                'button_id' => $buttonId,
            ]);

            // Find which flow this button belongs to by analyzing all active flows
            $targetFlow = $this->findFlowContainingButton($buttonId);

            if ($targetFlow) {
                whatsapp_log('Found flow containing button', 'info', [
                    'flow_id'   => $targetFlow->id,
                    'button_id' => $buttonId,
                ]);

                // Continue execution in the found flow
                return $this->continueFlowExecution(
                    $targetFlow,
                    $buttonId,
                    [], // Empty context - we don't need it
                    $contactData,
                    $triggerMsg,
                    $chatId,
                    $contactNumber,
                    $phoneNumberId
                );
            } else {
                whatsapp_log('No flow found containing button', 'warning', [
                    'button_id' => $buttonId,
                ]);
            }
        }

        // Handle new flow triggers (text messages)
        if ($triggerMsg && ! $buttonId) {
            whatsapp_log('Looking for new flow to trigger', 'info', [
                'trigger_msg' => $triggerMsg,
            ]);

            $flows = BotFlow::where('is_active', 1)->get();

            foreach ($flows as $flow) {
                $flowData = json_decode($flow->flow_data, true);
                if (empty($flowData) || empty($flowData['nodes'])) {
                    continue;
                }

                // Find trigger node
                foreach ($flowData['nodes'] as $node) {
                    if ($node['type'] === 'trigger') {
                        if ($this->isFlowMatch($node, $contactData->type, $triggerMsg)) {
                            whatsapp_log('Found matching trigger, executing flow', 'info', [
                                'flow_id'         => $flow->id,
                                'trigger_node_id' => $node['id'],
                            ]);

                            return $this->executeFlowFromStart($flow, $contactData, $triggerMsg, $chatId, $contactNumber, $phoneNumberId);
                        }
                    }
                }
            }
        }

        whatsapp_log('No flow execution determined', 'info');

        return false;
    }

    /**
     * Find which flow contains the button that was pressed
     */
    private function findFlowContainingButton($buttonId)
    {
        // Parse button ID to get source node
        $navigationInfo = $this->parseButtonIdForNavigation($buttonId);

        if (! $navigationInfo || ! $navigationInfo['source_node_id']) {
            whatsapp_log('Cannot determine source node from button ID', 'warning', [
                'button_id' => $buttonId,
            ]);

            return null;
        }

        $sourceNodeId = $navigationInfo['source_node_id'];

        whatsapp_log('Looking for flow containing source node', 'debug', [
            'source_node_id' => $sourceNodeId,
            'button_id'      => $buttonId,
        ]);

        // Search all active flows for the source node
        $flows = BotFlow::where('is_active', 1)->get();

        foreach ($flows as $flow) {
            $flowData = json_decode($flow->flow_data, true);
            if (empty($flowData) || empty($flowData['nodes'])) {
                continue;
            }

            // Check if this flow contains the source node
            foreach ($flowData['nodes'] as $node) {
                if ($node['id'] === $sourceNodeId) {
                    whatsapp_log('Found flow containing source node', 'info', [
                        'flow_id'        => $flow->id,
                        'source_node_id' => $sourceNodeId,
                        'node_type'      => $node['type'],
                    ]);

                    return $flow;
                }
            }
        }

        whatsapp_log('No flow found containing source node', 'warning', [
            'source_node_id' => $sourceNodeId,
        ]);

        return null;
    }

    private function findTargetFromFlowData($sourceNodeId, $buttonId, $flowData)
    {
        $edges = $flowData['edges'] ?? [];

        whatsapp_log('Finding target from flow data', 'debug', [
            'source_node' => $sourceNodeId,
            'button_id'   => $buttonId,
            'total_edges' => count($edges),
        ]);

        foreach ($edges as $edge) {
            if ($edge['source'] === $sourceNodeId) {
                $sourceHandle = $edge['sourceHandle'] ?? null;
                $target       = $edge['target'];

                whatsapp_log('Checking edge', 'debug', [
                    'edge_id'       => $edge['id'],
                    'source_handle' => $sourceHandle,
                    'target'        => $target,
                ]);

                if ($sourceHandle) {
                    // Handle unique button IDs: 1751267259695_btn_0 -> button-0
                    if (strpos($buttonId, '_btn_') !== false) {
                        $parts = explode('_btn_', $buttonId);
                        if (count($parts) === 2) {
                            $buttonIndex    = $parts[1];
                            $expectedHandle = 'button-' . $buttonIndex;

                            whatsapp_log('Comparing handles', 'debug', [
                                'button_index'    => $buttonIndex,
                                'expected_handle' => $expectedHandle,
                                'actual_handle'   => $sourceHandle,
                            ]);

                            if ($sourceHandle === $expectedHandle) {
                                whatsapp_log('Handle match found', 'info', [
                                    'button_id'     => $buttonId,
                                    'source_handle' => $sourceHandle,
                                    'target'        => $target,
                                ]);

                                return $target;
                            }
                        }
                    }

                    // Handle generic button IDs: button1 -> button-0
                    if (preg_match('/^button(\d+)$/', $buttonId, $matches)) {
                        $buttonNumber   = intval($matches[1]);
                        $expectedHandle = 'button-' . ($buttonNumber - 1);
                        if ($sourceHandle === $expectedHandle) {
                            return $target;
                        }
                    }
                } else {
                    // Default edge without specific handle - use for any button
                    whatsapp_log('Using default edge', 'debug', [
                        'target' => $target,
                    ]);

                    return $target;
                }
            }
        }

        whatsapp_log('No matching edge found', 'warning', [
            'source_node' => $sourceNodeId,
            'button_id'   => $buttonId,
        ]);

        return null;
    }

    private function parseButtonIdForNavigation($buttonId)
    {
        whatsapp_log('Parsing button/list ID', 'debug', [
            'button_id' => $buttonId,
        ]);

        // Format 1: Button ID - "1751267259695_btn_0"
        if (strpos($buttonId, '_btn_') !== false) {
            $parts = explode('_btn_', $buttonId);
            if (count($parts) === 2 && is_numeric($parts[1])) {
                return [
                    'source_node_id'   => $parts[0],
                    'button_index'     => intval($parts[1]),
                    'format'           => 'unique_button_id',
                    'interaction_type' => 'button',
                ];
            }
        }

        // Format 2: List Item ID - "1751281693715_item_0_0"
        if (strpos($buttonId, '_item_') !== false) {
            $parts = explode('_item_', $buttonId);
            if (count($parts) === 2) {
                $sourceNodeId = $parts[0];
                $itemParts    = explode('_', $parts[1]);
                if (count($itemParts) === 2 && is_numeric($itemParts[0]) && is_numeric($itemParts[1])) {
                    return [
                        'source_node_id'   => $sourceNodeId,
                        'section_index'    => intval($itemParts[0]),
                        'item_index'       => intval($itemParts[1]),
                        'format'           => 'unique_list_item_id',
                        'interaction_type' => 'list',
                    ];
                }
            }
        }

        // Format 3: Generic button ID - "button1", "button2", etc.
        if (preg_match('/^button(\d+)$/', $buttonId, $matches)) {
            $buttonNumber = intval($matches[1]);

            return [
                'source_node_id'   => null,
                'button_index'     => $buttonNumber - 1,
                'format'           => 'generic_button',
                'interaction_type' => 'button',
            ];
        }

        whatsapp_log('Unknown button/list ID format', 'warning', [
            'button_id' => $buttonId,
        ]);

        return null;
    }

    /**
     * Find target node by analyzing flow edges for specific button press
     */
    private function findTargetNodeFromButtonPress($sourceNodeId, $buttonIndex, $flowData, $interactionType = 'button', $sectionIndex = null, $itemIndex = null)
    {
        // This method now returns the FIRST target for backward compatibility
        $allTargets = $this->findAllTargetNodesFromButtonPress($sourceNodeId, $buttonIndex, $flowData, $interactionType, $sectionIndex, $itemIndex);

        if (! empty($allTargets)) {
            whatsapp_log('Using first target for backward compatibility', 'debug', [
                'first_target'  => $allTargets[0],
                'total_targets' => count($allTargets),
            ]);

            return $allTargets[0];
        }

        return null;
    }

    /**
     * Continue flow execution when user responds to buttons/lists
     * This method handles user interactions that should lead to next nodes
     */
    private function continueFlowExecution($flow, $buttonId, $flowContext, $contactData, $triggerMsg, $chatId, $contactNumber, $phoneNumberId)
    {
        whatsapp_log('Database-free flow continuation', 'info', [
            'flow_id'   => $flow->id,
            'button_id' => $buttonId,
        ]);

        // Parse button/list ID to extract navigation information
        $navigationInfo = $this->parseButtonIdForNavigation($buttonId);

        if (! $navigationInfo) {
            whatsapp_log('Cannot parse interaction ID for navigation', 'error', [
                'button_id' => $buttonId,
            ]);

            return false;
        }

        $sourceNodeId    = $navigationInfo['source_node_id'];
        $interactionType = $navigationInfo['interaction_type'];

        whatsapp_log('Parsed interaction navigation info', 'debug', [
            'source_node_id'   => $sourceNodeId,
            'interaction_type' => $interactionType,
            'navigation_info'  => $navigationInfo,
        ]);

        // Get flow data and find ALL target nodes
        $flowData = json_decode($flow->flow_data, true);

        if ($interactionType === 'button') {
            $targetNodeIds = $this->findAllTargetNodesFromButtonPress(
                $sourceNodeId,
                $navigationInfo['button_index'],
                $flowData,
                'button'
            );
        } elseif ($interactionType === 'list') {
            $targetNodeIds = $this->findAllTargetNodesFromButtonPress(
                $sourceNodeId,
                null,
                $flowData,
                'list',
                $navigationInfo['section_index'],
                $navigationInfo['item_index']
            );
        } else {
            $targetNodeIds = [];
        }

        if (empty($targetNodeIds)) {
            whatsapp_log('No target nodes found for interaction', 'warning', [
                'source_node_id'   => $sourceNodeId,
                'interaction_type' => $interactionType,
                'navigation_info'  => $navigationInfo,
            ]);

            return false;
        }

        whatsapp_log('Found target nodes for interaction', 'info', [
            'source_node_id'   => $sourceNodeId,
            'target_node_ids'  => $targetNodeIds,
            'interaction_type' => $interactionType,
            'total_targets'    => count($targetNodeIds),
        ]);

        // Get the actual node objects
        $targetNodes = $this->getTargetNodeObjects($targetNodeIds, $flowData);

        if (empty($targetNodes)) {
            whatsapp_log('Target node objects not found', 'error', [
                'target_node_ids' => $targetNodeIds,
            ]);

            return false;
        }

        // Process ALL target nodes using the existing sequential processing logic
        $context = [
            'flow_id'            => $flow->id,
            'chat_id'            => $chatId,
            'trigger_message'    => $triggerMsg,
            'is_button_response' => true,
        ];

        return $this->processConnectedNodesSequentially(
            $targetNodes,
            $flowData,
            $contactData,
            $triggerMsg,
            $chatId,
            $contactNumber,
            $phoneNumberId,
            $context
        );
    }

    /**
     * Find target node ID based on button ID and flow context
     */
    private function findTargetNodeFromButtonId($buttonId, $flowContext, $flowData)
    {
        // This method would need to implement logic to map button IDs to target nodes
        // The exact implementation depends on how you store button->node mappings

        // Example implementation for button messages:
        if (! empty($flowContext['current_node'])) {
            $sourceNodeId = $flowContext['current_node'];
            $edges        = $flowData['edges'] ?? [];

            // Look for edges that have sourceHandle matching the button
            foreach ($edges as $edge) {
                if ($edge['source'] === $sourceNodeId) {
                    // Check if this edge corresponds to the button clicked
                    // This logic might need adjustment based on how you store button mappings
                    if (isset($edge['sourceHandle']) && strpos($edge['sourceHandle'], $buttonId) !== false) {
                        return $edge['target'];
                    }

                    // If no specific handle, assume any edge from this node is valid
                    // (You might want to implement more specific logic here)
                    if (empty($edge['sourceHandle']) || $edge['sourceHandle'] === null) {
                        return $edge['target'];
                    }
                }
            }
        }

        return null;
    }

    /**
     * Execute flow starting from a specific node ID
     */
    private function executeFromSpecificNode($flow, $nodeId, $contactData, $triggerMsg, $chatId, $contactNumber, $phoneNumberId)
    {
        $flowData = json_decode($flow->flow_data, true);
        if (empty($flowData) || empty($flowData['nodes'])) {
            return false;
        }

        // Find the target node
        $targetNode = null;
        foreach ($flowData['nodes'] as $node) {
            if ($node['id'] === $nodeId) {
                $targetNode = $node;
                break;
            }
        }

        if (! $targetNode) {
            whatsapp_log('Target node not found', 'error', [
                'target_node_id' => $nodeId,
                'flow_id'        => $flow->id,
            ]);

            return false;
        }

        whatsapp_log('Executing from specific node', 'info', [
            'target_node_id' => $nodeId,
            'node_type'      => $targetNode['type'],
            'flow_id'        => $flow->id,
        ]);

        $context = [
            'flow_id'            => $flow->id,
            'chat_id'            => $chatId,
            'trigger_message'    => $triggerMsg,
            'current_node'       => $nodeId,
            'is_button_response' => true,
        ];

        // Process this single node as an array (for consistency with processConnectedNodesSequentially)
        return $this->processConnectedNodesSequentially(
            [$targetNode], // Single node in array
            $flowData,
            $contactData,
            $triggerMsg,
            $chatId,
            $contactNumber,
            $phoneNumberId,
            $context
        );
    }

    private function executeFlowWithMultipleTriggers($flow, $triggerNodes, $contactData, $triggerMsg, $chatId, $contactNumber, $phoneNumberId)
    {
        $flowData = json_decode($flow->flow_data, true);
        if (empty($flowData) || empty($flowData['nodes'])) {
            return false;
        }

        $context = [
            'flow_id'         => $flow->id,
            'chat_id'         => $chatId,
            'trigger_message' => $triggerMsg,
        ];

        $totalSuccessCount = 0;

        // Process each matching trigger
        foreach ($triggerNodes as $triggerNode) {
            whatsapp_log('Processing trigger node', 'debug', [
                'trigger_id'      => $triggerNode['id'],
                'flow_id'         => $flow->id,
                'trigger_message' => $triggerMsg,
            ]);

            // Find nodes connected to this specific trigger
            $connectedNodes = $this->findDirectlyConnectedNodes($triggerNode['id'], $flowData);

            if (! empty($connectedNodes)) {
                whatsapp_log('Found connected nodes for trigger', 'debug', [
                    'trigger_id'      => $triggerNode['id'],
                    'connected_count' => count($connectedNodes),
                    'connected_nodes' => array_map(function ($node) {
                        return ['id' => $node['id'], 'type' => $node['type']];
                    }, $connectedNodes),
                ]);

                // Process connected nodes for this trigger
                $result = $this->processConnectedNodesSequentially(
                    $connectedNodes,
                    $flowData,
                    $contactData,
                    $triggerMsg,
                    $chatId,
                    $contactNumber,
                    $phoneNumberId,
                    $context
                );

                if ($result) {
                    $totalSuccessCount++;
                }

                // Add delay between different trigger executions
                if (count($triggerNodes) > 1) {
                    usleep(800000); // 800ms delay between different triggers
                }
            } else {
                whatsapp_log('No connected nodes found for trigger', 'warning', [
                    'trigger_id' => $triggerNode['id'],
                    'flow_id'    => $flow->id,
                ]);
            }
        }

        whatsapp_log('Completed processing all matching triggers', 'info', [
            'total_triggers'      => count($triggerNodes),
            'successful_triggers' => $totalSuccessCount,
            'flow_id'             => $flow->id,
        ]);

        return $totalSuccessCount > 0;
    }

    /**
     * Execute multiple nodes connected to a node
     * - Modified to better handle node connections and execution order
     */
    private function executeMultipleNodes($flow, $nodes, $contactData, $triggerMsg, $chatId, $contactNumber, $phoneNumberId, $context)
    {
        $results  = [];
        $flowData = json_decode($flow->flow_data, true);

        // Update context to include flow data for node navigation
        $context['flow_id'] = $flow->id;
        $context['chat_id'] = $chatId;

        // First, order nodes by their position to ensure logical execution
        usort($nodes, function ($a, $b) {
            // Order by Y position first (vertical order)
            if ($a['position']['y'] != $b['position']['y']) {
                return $a['position']['y'] <=> $b['position']['y'];
            }

            // If same Y, order by X position (horizontal order)
            return $a['position']['x'] <=> $b['position']['x'];
        });

        foreach ($nodes as $node) {
            // Enhanced tracking of current node in the flow
            $context['current_node'] = $node['id'];

            $result = $this->processNode(
                $node,
                $flowData,
                $contactData,
                $triggerMsg,
                $chatId,
                $contactNumber,
                $phoneNumberId,
                $context
            );

            $results[] = $result;

            // Add small delay between messages to ensure proper order
            if (count($nodes) > 1) {
                usleep(200000); // 200ms delay
            }
        }

        // Return true if at least one node processed successfully
        return in_array(true, $results, true);
    }

    /**
     * Execute legacy message and template bots for backward compatibility
     */
    private function processLegacyBots($contactData, $triggerMsg, $interaction, $messageData)
    {
        $query_trigger_msg = $triggerMsg;
        $reply_type        = null;

        if ($this->is_first_time) {
            $query_trigger_msg = '';
            $reply_type        = 3;
        }

        // Fetch template and message bots based on interaction
        $template_bots = \App\Models\TemplateBot::getTemplateBotsByRelType($contactData->type ?? '', $query_trigger_msg, $reply_type);
        $message_bots  = \App\Models\MessageBots::getMessageBotsbyRelType($contactData->type ?? '', $query_trigger_msg, $reply_type);

        // If no match, try fallback response
        if (empty($template_bots) && empty($message_bots)) {
            $template_bots = \App\Models\TemplateBot::getTemplateBotsByRelType($contactData->type ?? '', $query_trigger_msg, 4);
            $message_bots  = \App\Models\MessageBots::getMessageBotsbyRelType($contactData->type ?? '', $query_trigger_msg, 4);
        }

        // Process templates
        $add_messages = function ($item) {
            $item['header_message'] = $item['header_data_text'];
            $item['body_message']   = $item['body_data'];
            $item['footer_message'] = $item['footer_data'];

            return $item;
        };

        $template_bots = array_map($add_messages, $template_bots);

        foreach ($template_bots as $template) {
            $template['rel_id'] = $contactData->id;
            if (! empty($contactData->userid)) {
                $template['userid'] = $contactData->userid;
            }

            // Send template on exact match, contains, or first time
            if ((1 == $template['reply_type'] && in_array(strtolower($triggerMsg), array_map('trim', array_map('strtolower', explode(',', $template['trigger']))))) || (2 == $template['reply_type'] && ! empty(array_filter(explode(',', strtolower($template['trigger'])), fn ($word) => preg_match('/\b' . preg_quote(trim($word), '/') . '\b/', strtolower($triggerMsg))))) || (3 == $template['reply_type'] && $this->is_first_time) || 4 == $template['reply_type']
            ) {

                $response = $this->sendTemplate(
                    $messageData['messages'][0]['from'],
                    $template,
                    'template_bot',
                    $messageData['metadata']['phone_number_id']
                );

                $this->storeBotMessages(
                    $template,
                    $interaction->id,
                    $contactData,
                    'template_bot',
                    $response
                );
            }
        }

        // Process message bots
        foreach ($message_bots as $message) {
            $message['rel_id'] = $contactData->id;
            if (! empty($contactData->userid)) {
                $message['userid'] = $contactData->userid;
            }

            if ((1 == $message['reply_type'] && in_array(strtolower($triggerMsg), array_map('trim', array_map('strtolower', explode(',', $message['trigger']))))) || (2 == $message['reply_type'] && ! empty(array_filter(explode(',', strtolower($message['trigger'])), fn ($word) => preg_match('/\b' . preg_quote(trim($word), '/') . '\b/', strtolower($triggerMsg))))) || (3 == $message['reply_type'] && $this->is_first_time) || 4 == $message['reply_type']
            ) {

                $response = $this->sendMessage(
                    $messageData['messages'][0]['from'],
                    $message,
                    $messageData['metadata']['phone_number_id']
                );

                $this->storeBotMessages(
                    $message,
                    $interaction->id,
                    $contactData,
                    '',
                    $response
                );
            }
        }
    }

    /**
     * Execute flow starting from a specific node ID
     */
    private function executeFlowFromNode($flow, $nodeId, $contactData, $triggerMsg, $chatId, $contactNumber, $phoneNumberId, $context = [])
    {
        $flowData = json_decode($flow->flow_data, true);
        if (empty($flowData) || empty($flowData['nodes'])) {
            return false;
        }

        // Find the node with the specified ID
        $startNode = null;
        foreach ($flowData['nodes'] as $node) {
            if ($node['id'] === $nodeId) {
                $startNode = $node;
                break;
            }
        }

        if (! $startNode) {
            return false;
        }

        // Set up execution context
        $context['flow_id']         = $flow->id;
        $context['chat_id']         = $chatId;
        $context['trigger_message'] = $triggerMsg;

        // Execute the flow starting from this node
        return $this->processNode(
            $startNode,
            $flowData,
            $contactData,
            $triggerMsg,
            $chatId,
            $contactNumber,
            $phoneNumberId,
            $context
        );
    }

    private function findDirectlyConnectedNodes($sourceNodeId, $flowData)
    {
        $connectedNodes = [];
        $edges          = $flowData['edges'] ?? [];
        $nodes          = $flowData['nodes'] ?? [];

        // Create a lookup map for nodes by ID
        $nodeMap = [];
        foreach ($nodes as $node) {
            $nodeMap[$node['id']] = $node;
        }

        // Find all edges that start from the source node
        foreach ($edges as $edge) {
            if ($edge['source'] === $sourceNodeId) {
                $targetNodeId = $edge['target'];
                if (isset($nodeMap[$targetNodeId])) {
                    $connectedNodes[] = $nodeMap[$targetNodeId];
                }
            }
        }

        whatsapp_log('Found directly connected nodes', 'debug', [
            'source_node'     => $sourceNodeId,
            'connected_count' => count($connectedNodes),
            'connected_nodes' => array_map(function ($node) {
                return ['id' => $node['id'], 'type' => $node['type']];
            }, $connectedNodes),
        ]);

        return $connectedNodes;
    }

    /**
     * Execute flow from the starting trigger node
     */
    private function executeFlowFromStart($flow, $contactData, $triggerMsg, $chatId, $contactNumber, $phoneNumberId)
    {
        $flowData = json_decode($flow->flow_data, true);
        if (empty($flowData) || empty($flowData['nodes'])) {
            return false;
        }

        // Find ALL matching trigger nodes (not just the first one)
        $matchingTriggers = [];
        foreach ($flowData['nodes'] as $node) {
            if ($node['type'] === 'trigger') {
                if ($this->isFlowMatch($node, $contactData->type, $triggerMsg)) {
                    $matchingTriggers[] = $node;
                }
            }
        }

        if (empty($matchingTriggers)) {
            whatsapp_log('No matching triggers found', 'warning', [
                'flow_id'         => $flow->id,
                'trigger_message' => $triggerMsg,
            ]);

            return false;
        }

        // Use the new method to handle multiple triggers
        return $this->executeFlowWithMultipleTriggers($flow, $matchingTriggers, $contactData, $triggerMsg, $chatId, $contactNumber, $phoneNumberId);
    }

    private function findAllConnectedNodes($sourceNodeId, $flowData)
    {
        $connectedNodes = [];
        $edges          = $flowData['edges'] ?? [];
        $nodes          = $flowData['nodes'] ?? [];

        // Create a lookup map for nodes by ID
        $nodeMap = [];
        foreach ($nodes as $node) {
            $nodeMap[$node['id']] = $node;
        }

        // Find all edges that start from the source node
        foreach ($edges as $edge) {
            if ($edge['source'] === $sourceNodeId) {
                $targetNodeId = $edge['target'];
                if (isset($nodeMap[$targetNodeId])) {
                    $connectedNodes[] = $nodeMap[$targetNodeId];
                }
            }
        }

        whatsapp_log('Found connected nodes', 'debug', [
            'source_node'     => $sourceNodeId,
            'connected_count' => count($connectedNodes),
            'connected_types' => array_map(function ($node) {
                return $node['type'];
            }, $connectedNodes),
        ]);

        return $connectedNodes;
    }

    private function processSingleNode($node, $contactData, $triggerMsg, $chatId, $contactNumber, $phoneNumberId, $context)
    {
        $nodeType = $node['type'];
        $nodeData = $node['data'] ?? [];

        // Skip trigger nodes as they don't send messages
        if ($nodeType === 'trigger') {
            return true;
        }

        whatsapp_log('Processing single node', 'info', [
            'node_id'        => $node['id'],
            'node_type'      => $nodeType,
            'contact_number' => $contactNumber,
            'chat_id'        => $chatId,
            'is_interactive' => in_array($nodeType, ['buttonMessage', 'listMessage']),
        ]);

        try {
            // Use the WhatsApp trait methods directly
            $result = $this->sendFlowMessage(
                $contactNumber,
                $nodeData,
                $nodeType,
                $phoneNumberId,
                $contactData,
                $context
            );

            whatsapp_log('sendFlowMessage result', 'debug', [
                'node_id'       => $node['id'],
                'node_type'     => $nodeType,
                'result_status' => $result['status'] ?? 'unknown',
                'has_data'      => isset($result['data']),
                'response_code' => $result['responseCode'] ?? 'unknown',
            ]);

            // Check if sending was successful
            if ($result && isset($result['status']) && $result['status']) {
                // Store the message
                $storeResult = $this->storeFlowMessage($result, $chatId, $contactNumber, $contactData, $nodeType, $nodeData, $context);

                whatsapp_log('Node processed and stored successfully', 'info', [
                    'node_id'         => $node['id'],
                    'node_type'       => $nodeType,
                    'storage_success' => $storeResult,
                    'chat_id'         => $chatId,
                ]);

                return true;
            } else {
                whatsapp_log('Node processing failed - sendFlowMessage returned false', 'error', [
                    'node_id'   => $node['id'],
                    'node_type' => $nodeType,
                    'result'    => $result,
                ]);

                return false;
            }
        } catch (\Exception $e) {
            whatsapp_log('Exception in processSingleNode', 'error', [
                'node_id'   => $node['id'],
                'node_type' => $nodeType,
                'error'     => $e->getMessage(),
                'trace'     => $e->getTraceAsString(),
            ], $e);

            return false;
        }
    }

    private function buildUniqueButtonMappings($buttons, $nextNodes)
    {
        $mappings = [];
        foreach ($buttons as $index => $button) {
            $buttonId = $button->getId();
            if (isset($nextNodes['button' . ($index + 1)])) {
                $mappings[$buttonId] = $nextNodes['button' . ($index + 1)];
            }
        }

        return $mappings;
    }

    private function buildButtonTargetMappings($sourceNodeId, $flowData)
    {
        $mappings = [];
        $edges    = $flowData['edges'] ?? [];

        whatsapp_log('Building button mappings for node', 'debug', [
            'source_node' => $sourceNodeId,
            'total_edges' => count($edges),
        ]);

        foreach ($edges as $edge) {
            if ($edge['source'] === $sourceNodeId) {
                $targetNodeId = $edge['target'];
                $sourceHandle = $edge['sourceHandle'] ?? null;

                whatsapp_log('Found edge from source node', 'debug', [
                    'edge_id'      => $edge['id'],
                    'source'       => $edge['source'],
                    'target'       => $targetNodeId,
                    'sourceHandle' => $sourceHandle,
                ]);

                if ($sourceHandle) {
                    // Handle button-0, button-1, etc.
                    if (preg_match('/button-(\d+)/', $sourceHandle, $matches)) {
                        $buttonIndex = $matches[1];

                        // Create multiple mapping formats for compatibility
                        $uniqueButtonId  = $sourceNodeId . '_btn_' . $buttonIndex;
                        $genericButtonId = 'button' . ($buttonIndex + 1);

                        $mappings[$uniqueButtonId]  = $targetNodeId;
                        $mappings[$genericButtonId] = $targetNodeId;

                        whatsapp_log('Added button mapping', 'debug', [
                            'unique_id'  => $uniqueButtonId,
                            'generic_id' => $genericButtonId,
                            'target'     => $targetNodeId,
                        ]);
                    }
                } else {
                    // Default edge without specific handle
                    $mappings['default'] = $targetNodeId;
                    whatsapp_log('Added default mapping', 'debug', [
                        'target' => $targetNodeId,
                    ]);
                }
            }
        }

        whatsapp_log('Final button mappings built', 'info', [
            'source_node' => $sourceNodeId,
            'mappings'    => $mappings,
        ]);

        return $mappings;
    }

    private function processConnectedNodesSequentially($nodes, $flowData, $contactData, $triggerMsg, $chatId, $contactNumber, $phoneNumberId, $context)
    {
        if (empty($nodes)) {
            return false;
        }

        whatsapp_log('Processing connected nodes sequentially', 'info', [
            'total_nodes' => count($nodes),
            'node_types'  => array_map(function ($node) {
                return $node['type'];
            }, $nodes),
            'chat_id' => $chatId,
        ]);

        // Sort nodes by their position for logical execution order
        usort($nodes, function ($a, $b) {
            if ($a['position']['y'] != $b['position']['y']) {
                return $a['position']['y'] <=> $b['position']['y'];
            }

            return $a['position']['x'] <=> $b['position']['x'];
        });

        $successCount             = 0;
        $stoppedAtInteractiveNode = false;

        foreach ($nodes as $index => $node) {
            try {
                $nodeType = $node['type'];

                whatsapp_log('Processing node in sequence', 'debug', [
                    'node_id'   => $node['id'],
                    'node_type' => $nodeType,
                    'position'  => $node['position'],
                    'index'     => $index + 1,
                    'total'     => count($nodes),
                ]);

                // Build context for this node
                $nodeContext = array_merge($context, [
                    'current_node'      => $node['id'],
                    'sequence_position' => $index,
                    'flow_id'           => $context['flow_id'],
                ]);

                // Check if this is an interactive node that should stop the sequence
                $isInteractiveNode = in_array($nodeType, ['buttonMessage', 'listMessage']);

                if ($isInteractiveNode) {
                    whatsapp_log('Found interactive node - building target mappings', 'info', [
                        'node_id'              => $node['id'],
                        'node_type'            => $nodeType,
                        'will_stop_after_this' => true,
                    ]);

                    // Build target mappings for interactive nodes
                    if ($nodeType === 'buttonMessage') {
                        $targetMappings            = $this->buildButtonTargetMappings($node['id'], $flowData);
                        $nodeContext['next_nodes'] = $targetMappings;
                    } elseif ($nodeType === 'listMessage') {
                        $targetMappings            = $this->buildListTargetMappings($node['id'], $flowData);
                        $nodeContext['next_nodes'] = $targetMappings;
                    }

                    whatsapp_log('Built target mappings for interactive node', 'debug', [
                        'node_id'         => $node['id'],
                        'node_type'       => $nodeType,
                        'target_mappings' => $targetMappings ?? [],
                        'mappings_count'  => count($targetMappings ?? []),
                    ]);
                }

                // Process the node
                $result = $this->processSingleNode($node, $contactData, $triggerMsg, $chatId, $contactNumber, $phoneNumberId, $nodeContext);

                if ($result) {
                    $successCount++;

                    if ($isInteractiveNode) {
                        whatsapp_log('Interactive node processed successfully - STOPPING SEQUENCE', 'info', [
                            'node_id'         => $node['id'],
                            'node_type'       => $nodeType,
                            'remaining_nodes' => count($nodes) - ($index + 1),
                            'reason'          => 'waiting_for_user_interaction',
                        ]);

                        $stoppedAtInteractiveNode = true;
                        break; // CRITICAL: Stop processing here for interactive nodes
                    } else {
                        whatsapp_log('Non-interactive node processed successfully', 'info', [
                            'node_id'   => $node['id'],
                            'node_type' => $nodeType,
                        ]);
                    }
                } else {
                    whatsapp_log('Node processing failed', 'warning', [
                        'node_id'        => $node['id'],
                        'node_type'      => $nodeType,
                        'is_interactive' => $isInteractiveNode,
                    ]);
                }

                // Add delay between non-interactive messages only
                if (! $isInteractiveNode && $index < count($nodes) - 1) {
                    usleep(500000); // 500ms delay
                }
            } catch (\Exception $e) {
                whatsapp_log('Error processing node in sequence', 'error', [
                    'node_id'   => $node['id'],
                    'node_type' => $node['type'],
                    'error'     => $e->getMessage(),
                    'trace'     => $e->getTraceAsString(),
                ], $e);
            }
        }

        whatsapp_log('Completed processing connected nodes', 'info', [
            'total_nodes'            => count($nodes),
            'successful_nodes'       => $successCount,
            'stopped_at_interactive' => $stoppedAtInteractiveNode,
            'processed_all'          => ! $stoppedAtInteractiveNode && $successCount === count($nodes),
        ]);

        return $successCount > 0;
    }

    private function findAllTargetNodesFromButtonPress($sourceNodeId, $buttonIndex, $flowData, $interactionType = 'button', $sectionIndex = null, $itemIndex = null)
    {
        $edges       = $flowData['edges'] ?? [];
        $targetNodes = []; // Changed to array to collect ALL targets

        whatsapp_log('Finding all targets from interaction', 'debug', [
            'source_node_id'   => $sourceNodeId,
            'interaction_type' => $interactionType,
            'button_index'     => $buttonIndex,
            'section_index'    => $sectionIndex,
            'item_index'       => $itemIndex,
            'total_edges'      => count($edges),
        ]);

        foreach ($edges as $edge) {
            if ($edge['source'] === $sourceNodeId) {
                $sourceHandle = $edge['sourceHandle'] ?? null;
                $target       = $edge['target'];

                whatsapp_log('Checking edge for interaction match', 'debug', [
                    'edge_id'       => $edge['id'],
                    'source_handle' => $sourceHandle,
                    'target'        => $target,
                ]);

                $isMatch = false;

                if ($interactionType === 'button') {
                    // Match button handles: "button-0", "button-1", etc.
                    $expectedHandle = 'button-' . $buttonIndex;
                    if ($sourceHandle === $expectedHandle) {
                        $isMatch = true;
                    }
                } elseif ($interactionType === 'list') {
                    // Match list item handles: "item-0-0", "item-1-2", etc.
                    $expectedHandle = 'item-' . $sectionIndex . '-' . $itemIndex;
                    if ($sourceHandle === $expectedHandle) {
                        $isMatch = true;
                    }
                }

                // Fallback: if no specific handle and this is the first interaction
                if (! $isMatch && ! $sourceHandle && ($buttonIndex === 0 || ($sectionIndex === 0 && $itemIndex === 0))) {
                    whatsapp_log('Using fallback edge for first interaction', 'debug', [
                        'target' => $target,
                    ]);
                    $isMatch = true;
                }

                if ($isMatch) {
                    $targetNodes[] = $target;
                    whatsapp_log('Found matching target for interaction', 'info', [
                        'source_node'      => $sourceNodeId,
                        'target_node'      => $target,
                        'interaction_type' => $interactionType,
                        'handle'           => $sourceHandle,
                    ]);
                }
            }
        }

        whatsapp_log('Found all targets for interaction', 'info', [
            'source_node_id'   => $sourceNodeId,
            'interaction_type' => $interactionType,
            'total_targets'    => count($targetNodes),
            'target_nodes'     => $targetNodes,
        ]);

        return $targetNodes;
    }

    private function getTargetNodeObjects($targetNodeIds, $flowData)
    {
        $nodes       = $flowData['nodes'] ?? [];
        $targetNodes = [];

        foreach ($targetNodeIds as $targetNodeId) {
            foreach ($nodes as $node) {
                if ($node['id'] === $targetNodeId) {
                    $targetNodes[] = $node;
                    break;
                }
            }
        }

        whatsapp_log('Retrieved target node objects', 'debug', [
            'target_ids'  => $targetNodeIds,
            'found_nodes' => count($targetNodes),
            'node_types'  => array_map(function ($node) {
                return $node['type'];
            }, $targetNodes),
        ]);

        return $targetNodes;
    }

    private function processAllConnectedNodes($flow, $nodes, $flowData, $contactData, $triggerMsg, $chatId, $contactNumber, $phoneNumberId, $context)
    {
        if (empty($nodes)) {
            return false;
        }

        whatsapp_log('Processing all connected nodes', 'debug', [
            'nodeCount' => count($nodes),
            'nodeTypes' => array_map(function ($node) {
                return $node['type'];
            }, $nodes),
            'chatId' => $chatId,
        ]);

        // Sort nodes by their position to ensure logical execution order
        usort($nodes, function ($a, $b) {
            // Order by Y position first (vertical order)
            if ($a['position']['y'] != $b['position']['y']) {
                return $a['position']['y'] <=> $b['position']['y'];
            }

            // If same Y, order by X position (horizontal order)
            return $a['position']['x'] <=> $b['position']['x'];
        });

        $processedCount = 0;
        $successCount   = 0;

        // Process each node one by one with proper delay between messages
        foreach ($nodes as $index => $node) {
            try {
                // Update context for current node
                $context['current_node'] = $node['id'];
                $context['node_index']   = $index;
                $context['total_nodes']  = count($nodes);

                whatsapp_log('Processing node', 'debug', [
                    'node_id'     => $node['id'],
                    'node_type'   => $node['type'],
                    'node_index'  => $index + 1,
                    'total_nodes' => count($nodes),
                ]);

                $result = $this->processNode(
                    $node,
                    $flowData,
                    $contactData,
                    $triggerMsg,
                    $chatId,
                    $contactNumber,
                    $phoneNumberId,
                    $context
                );

                $processedCount++;

                if ($result) {
                    $successCount++;
                    whatsapp_log('Node processed successfully', 'info', [
                        'node_id'   => $node['id'],
                        'node_type' => $node['type'],
                    ]);
                } else {
                    whatsapp_log('Node processing failed', 'warning', [
                        'node_id'   => $node['id'],
                        'node_type' => $node['type'],
                    ]);
                }

                // Add delay between messages to ensure proper delivery order
                if ($index < count($nodes) - 1) {
                    usleep(300000); // 300ms delay between messages
                }
            } catch (\Exception $e) {
                whatsapp_log('Error processing node', 'error', [
                    'node_id'   => $node['id'],
                    'node_type' => $node['type'],
                    'error'     => $e->getMessage(),
                ], $e);
            }
        }

        whatsapp_log('Completed processing all nodes', 'info', [
            'total_nodes'     => count($nodes),
            'processed_count' => $processedCount,
            'success_count'   => $successCount,
        ]);

        // Return true if at least one node processed successfully
        return $successCount > 0;
    }

    private function buildListTargetMappings($sourceNodeId, $flowData)
    {
        $mappings = [];
        $edges    = $flowData['edges'] ?? [];

        whatsapp_log('Building list mappings for node', 'debug', [
            'source_node' => $sourceNodeId,
            'total_edges' => count($edges),
        ]);

        foreach ($edges as $edge) {
            if ($edge['source'] === $sourceNodeId) {
                $targetNodeId = $edge['target'];
                $sourceHandle = $edge['sourceHandle'] ?? null;

                whatsapp_log('Found edge from list source node', 'debug', [
                    'edge_id'      => $edge['id'],
                    'source'       => $edge['source'],
                    'target'       => $targetNodeId,
                    'sourceHandle' => $sourceHandle,
                ]);

                if ($sourceHandle) {
                    // Handle list item handles: "item-0-0", "item-1-2", etc.
                    if (preg_match('/item-(\d+)-(\d+)/', $sourceHandle, $matches)) {
                        $sectionIndex = $matches[1];
                        $itemIndex    = $matches[2];

                        // Create mapping format for list items
                        $listItemId            = $sourceNodeId . '_item_' . $sectionIndex . '_' . $itemIndex;
                        $mappings[$listItemId] = $targetNodeId;

                        whatsapp_log('Added list item mapping', 'debug', [
                            'list_item_id'  => $listItemId,
                            'section_index' => $sectionIndex,
                            'item_index'    => $itemIndex,
                            'target'        => $targetNodeId,
                        ]);
                    }
                } else {
                    // Default edge without specific handle
                    $mappings['default'] = $targetNodeId;
                    whatsapp_log('Added default list mapping', 'debug', [
                        'target' => $targetNodeId,
                    ]);
                }
            }
        }

        whatsapp_log('Final list mappings built', 'info', [
            'source_node' => $sourceNodeId,
            'mappings'    => $mappings,
        ]);

        return $mappings;
    }

    // Updated findAllNextNodes to better handle edge sourceHandles
    private function findAllNextNodes($nodeId, $flowData)
    {
        if (empty($flowData['edges'])) {
            return [];
        }

        $connectedNodes = [];

        // Find all edges that start from this node
        $connectedEdges = [];
        foreach ($flowData['edges'] as $edge) {
            if ($edge['source'] === $nodeId) {
                $connectedEdges[] = $edge;
            }
        }

        // Find all nodes these edges point to
        foreach ($connectedEdges as $edge) {
            $targetNodeId = $edge['target'];
            foreach ($flowData['nodes'] as $node) {
                if ($node['id'] === $targetNodeId) {
                    // Add the sourceHandle to the node data for context
                    if (isset($edge['sourceHandle'])) {
                        $node['_sourceHandle'] = $edge['sourceHandle'];
                    }
                    $connectedNodes[] = $node;
                    break;
                }
            }
        }

        return $connectedNodes;
    }

    /**
     * Find the next node connected to a node ID
     */
    private function findNextNode($nodeId, $flowData)
    {
        if (empty($flowData['edges'])) {
            return null;
        }

        // Find the edge that starts from this node
        $nextEdge = null;
        foreach ($flowData['edges'] as $edge) {
            if ($edge['source'] === $nodeId && (! isset($edge['sourceHandle']) || $edge['sourceHandle'] === null)) {
                $nextEdge = $edge;
                break;
            }
        }

        if (! $nextEdge) {
            return null;
        }

        // Find the node this edge points to
        $nextNodeId = $nextEdge['target'];
        foreach ($flowData['nodes'] as $node) {
            if ($node['id'] === $nextNodeId) {
                return $node;
            }
        }

        return null;
    }

    /**
     * Enhanced processNode that's simplified since we now use processNodesInSequence
     * We keep this for backward compatibility but it's not the main processing method anymore
     */
    private function processNode($node, $flowData, $contactData, $triggerMsg, $chatId, $contactNumber, $phoneNumberId, $context)
    {
        try {
            $nodeType = $node['type'];
            $nodeData = $node['data'] ?? [];

            whatsapp_log('Processing individual node', 'debug', [
                'node_id'    => $node['id'],
                'node_type'  => $nodeType,
                'has_output' => ! empty($nodeData['output']),
            ]);

            // Skip trigger nodes as they don't send messages
            if ($nodeType === 'trigger') {
                return true;
            }

            $result = $this->sendFlowMessage(
                $contactNumber,
                $nodeData,
                $nodeType,
                $phoneNumberId,
                $contactData,
                $context
            );

            if ($result && isset($result['status']) && $result['status']) {
                whatsapp_log('Flow message sent successfully', 'info', [
                    'node_id'   => $node['id'],
                    'node_type' => $nodeType,
                    'to'        => $contactNumber,
                ]);

                return true;
            } else {
                whatsapp_log('Flow message failed', 'error', [
                    'node_id'   => $node['id'],
                    'node_type' => $nodeType,
                    'result'    => $result,
                ]);

                return false;
            }
        } catch (\Exception $e) {
            whatsapp_log('Exception in processNode', 'error', [
                'node_id' => $node['id'] ?? 'unknown',
                'error'   => $e->getMessage(),
            ], $e);

            return false;
        }
    }

    /**
     * Count outgoing edges from a node
     */
    private function countOutgoingEdges($nodeId, $flowData)
    {
        $count = 0;

        foreach ($flowData['edges'] as $edge) {
            if ($edge['source'] === $nodeId) {
                $count++;
            }
        }

        return $count;
    }

    /**
     * Check if a node is directly connected to a trigger node
     */
    private function isDirectChildOfTrigger($nodeId, $flowData)
    {
        // Find trigger node
        $triggerNodeId = null;

        foreach ($flowData['nodes'] as $node) {
            if ($node['type'] === 'trigger') {
                $triggerNodeId = $node['id'];
                break;
            }
        }

        if (! $triggerNodeId) {
            return false;
        }

        // Check if there's an edge from trigger to this node
        foreach ($flowData['edges'] as $edge) {
            if ($edge['source'] === $triggerNodeId && $edge['target'] === $nodeId) {
                return true;
            }
        }

        return false;
    }

    /**
     * Find the next node for a condition based on result
     */
    private function findNextNodeForCondition($nodeId, $flowData, $conditionResult)
    {
        if (empty($flowData['edges'])) {
            return null;
        }

        // Find the edge that matches condition result
        $nextEdge = null;
        foreach ($flowData['edges'] as $edge) {
            if ($edge['source'] === $nodeId) {
                if (($conditionResult && $edge['sourceHandle'] === 'yes') || (! $conditionResult && $edge['sourceHandle'] === 'no')
                ) {
                    $nextEdge = $edge;
                    break;
                }
            }
        }

        if (! $nextEdge) {
            return null;
        }

        // Find the node this edge points to
        $nextNodeId = $nextEdge['target'];
        foreach ($flowData['nodes'] as $node) {
            if ($node['id'] === $nextNodeId) {
                return $node;
            }
        }

        return null;
    }

    /**
     * Improved findButtonTargets method that captures ALL connections
     */
    private function findButtonTargets($nodeId, $flowData)
    {
        if (empty($flowData['edges'])) {
            return [];
        }

        $targets = [];

        foreach ($flowData['edges'] as $edge) {
            if (
                $edge['source'] === $nodeId && isset($edge['sourceHandle']) && (strpos($edge['sourceHandle'], 'button-') === 0)
            ) {
                // Extract button index from the sourceHandle (e.g., "button-0" → "0")
                $buttonIndex = str_replace('button-', '', $edge['sourceHandle']);

                // Find the source node to get the button data
                $sourceNode = null;
                foreach ($flowData['nodes'] as $node) {
                    if ($node['id'] === $nodeId) {
                        $sourceNode = $node;
                        break;
                    }
                }

                if ($sourceNode && isset($sourceNode['data']['buttons'][$buttonIndex])) {
                    $buttonValue = $sourceNode['data']['buttons'][$buttonIndex]['value'];

                    // Store the target node ID mapped to the button value
                    $targets[$buttonValue] = $edge['target'];

                    // LOG for debugging
                    whatsapp_log('Mapped button to target', 'debug', [
                        'buttonIndex' => $buttonIndex,
                        'buttonValue' => $buttonValue,
                        'targetNode'  => $edge['target'],
                    ]);
                }
            }
        }

        return $targets;
    }

    /**
     * Add this new method to process all nodes in sequence
     * This is the key method for ensuring all connected nodes are processed
     */
    private function processNodesInSequence($flow, $nodes, $flowData, $contactData, $triggerMsg, $chatId, $contactNumber, $phoneNumberId, $context)
    {
        if (empty($nodes)) {
            return false;
        }

        // LOG for debugging
        whatsapp_log('Processing nodes in sequence', 'debug', [
            'nodeCount' => count($nodes),
            'nodeTypes' => array_map(function ($node) {
                return $node['type'];
            }, $nodes),
            'chatId' => $chatId,
        ]);

        // Sort nodes by their position to ensure logical execution
        usort($nodes, function ($a, $b) {
            // Order by Y position first (vertical order)
            if ($a['position']['y'] != $b['position']['y']) {
                return $a['position']['y'] <=> $b['position']['y'];
            }

            // If same Y, order by X position (horizontal order)
            return $a['position']['x'] <=> $b['position']['x'];
        });

        $processedCount = 0;
        $success        = false;

        // Process each node one by one with proper delay between messages
        foreach ($nodes as $index => $node) {
            // Skip interactive nodes after the first one as they need user input
            // We only want to process non-interactive nodes automatically
            if ($index > 0 && in_array($node['type'], ['buttonMessage', 'listMessage'])) {
                // Log that we're skipping an interactive node
                whatsapp_log('Skipping interactive node in sequence', 'debug', [
                    'nodeType' => $node['type'],
                    'nodeId'   => $node['id'],
                ]);

                continue;
            }

            // Update context for this node
            $nodeContext = array_merge($context, [
                'current_node'      => $node['id'],
                'sequence_position' => $index,
                'flow_id'           => $flow->id,
            ]);

            // Process the node
            $nodeType = $node['type'];
            $nodeData = $node['data'];
            $nodeId   = $node['id'];

            // For button and list messages, calculate and add all possible targets to context
            if ($nodeType === 'buttonMessage') {
                $buttonTargets             = $this->findButtonTargets($nodeId, $flowData);
                $nodeContext['next_nodes'] = $buttonTargets;
            } elseif ($nodeType === 'listMessage') {
                $listTargets               = $this->findListTargets($nodeId, $flowData);
                $nodeContext['next_nodes'] = $listTargets;
            }

            // Send the message based on node type
            $result = $this->sendFlowMessage(
                $contactNumber,
                $nodeData,
                $nodeType,
                $phoneNumberId,
                $contactData,
                $nodeContext
            );

            // Store the message with context for future reference
            $this->storeFlowMessage(
                $result,
                $chatId,
                $contactNumber,
                $contactData,
                $nodeType,
                $nodeData,
                $nodeContext
            );

            $processedCount++;
            $success = true;

            // Add small delay between messages to ensure proper order
            if (count($nodes) > 1 && $index < count($nodes) - 1) {
                usleep(300000); // 300ms delay between messages
            }
        }

        // LOG result
        whatsapp_log('Finished processing nodes in sequence', 'debug', [
            'processedCount' => $processedCount,
            'success'        => $success,
        ]);

        return $success;
    }

    /**
     * Enhanced findListTargets method to better map list items to target nodes
     */
    private function findListTargets($nodeId, $flowData)
    {
        if (empty($flowData['edges'])) {
            return [];
        }

        $targets = [];

        foreach ($flowData['edges'] as $edge) {
            if (
                $edge['source'] === $nodeId && isset($edge['sourceHandle']) && (strpos($edge['sourceHandle'], 'item-') === 0)
            ) {
                // Extract section and item indices from the sourceHandle (e.g., "item-0-1" → section 0, item 1)
                $parts = explode('-', $edge['sourceHandle']);
                if (count($parts) >= 3) {
                    $sectionIndex = $parts[1];
                    $itemIndex    = $parts[2];

                    // Find the source node to get the list data
                    $sourceNode = null;
                    foreach ($flowData['nodes'] as $node) {
                        if ($node['id'] === $nodeId) {
                            $sourceNode = $node;
                            break;
                        }
                    }

                    if (
                        $sourceNode && isset($sourceNode['data']['sections']) && isset($sourceNode['data']['sections'][$sectionIndex]) && isset($sourceNode['data']['sections'][$sectionIndex]['items']) && isset($sourceNode['data']['sections'][$sectionIndex]['items'][$itemIndex])
                    ) {
                        // Use both description and ID as potential keys
                        // This is because WhatsApp may return either one depending on the client
                        $itemData = $sourceNode['data']['sections'][$sectionIndex]['items'][$itemIndex];

                        // Primary key is the description if available
                        if (! empty($itemData['description'])) {
                            $targets[$itemData['description']] = $edge['target'];
                        }

                        // Also map by ID as fallback
                        $targets[$itemData['id']] = $edge['target'];

                        // And map by title as another fallback
                        if (! empty($itemData['title'])) {
                            $targets[$itemData['title']] = $edge['target'];
                        }

                        // LOG for debugging
                        whatsapp_log('Mapped list item to target', 'debug', [
                            'sectionIndex' => $sectionIndex,
                            'itemIndex'    => $itemIndex,
                            'itemDesc'     => $itemData['description'] ?? '',
                            'itemTitle'    => $itemData['title']       ?? '',
                            'targetNode'   => $edge['target'],
                        ]);
                    }
                }
            }
        }

        return $targets;
    }

    /**
     * Store a flow message in the chat history
     */
    private function storeFlowMessage($result, $chatId, $contactNumber, $contactData, $nodeType, $nodeData, $context)
    {
        try {
            whatsapp_log('Storing flow message in chat system', 'debug', [
                'chat_id'        => $chatId,
                'node_type'      => $nodeType,
                'contact_number' => $contactNumber,
            ]);

            // Get message ID from WhatsApp response
            $messageId = $this->extractMessageIdFromResult($result);

            if (! $messageId) {
                $messageId = uniqid('flow_msg_' . $nodeType . '_');
            }

            // Build the HTML content for chat display using existing pattern
            $messageHtml = $this->buildFlowMessageHtml($nodeType, $nodeData, $contactData, $context);

            // Extract plain text for last_message updates
            $plainTextMessage = $this->extractPlainTextFromFlowMessage($nodeType, $nodeData, $contactData);

            // Store in chat_messages table using existing structure
            $chat_message = [
                'interaction_id' => $chatId,
                'sender_id'      => get_setting('whatsapp.wm_default_phone_number', 'bot'),
                'url'            => $this->extractMediaUrl($nodeType, $nodeData),
                'message'        => $messageHtml,
                'status'         => 'sent',
                'time_sent'      => now()->toDateTimeString(),
                'message_id'     => $messageId,
                'staff_id'       => 0,
                'type'           => $this->mapNodeTypeToMessageType($nodeType),
                'is_read'        => 0,
                'ref_message_id' => null, // Flow messages don't reference other messages
            ];

            $message_db_id = ChatMessage::insertGetId($chat_message);

            // Update chat table with last message info (same as existing pattern)
            $this->updateChatLastMessage($chatId, $plainTextMessage);

            // Trigger Pusher notification (same as existing pattern)
            $this->triggerChatNotification($chatId, $message_db_id);

            whatsapp_log('Flow message stored successfully in chat system', 'info', [
                'chat_id'            => $chatId,
                'message_id'         => $messageId,
                'db_message_id'      => $message_db_id,
                'node_type'          => $nodeType,
                'plain_text_preview' => substr($plainTextMessage, 0, 50),
            ]);

            return true;
        } catch (\Exception $e) {
            whatsapp_log('Error storing flow message in chat system', 'error', [
                'error'     => $e->getMessage(),
                'chat_id'   => $chatId,
                'node_type' => $nodeType,
                'trace'     => $e->getTraceAsString(),
            ], $e);

            return false;
        }
    }

    private function extractMessageIdFromResult($result)
    {
        $messageId = null;

        try {
            if (isset($result['data'])) {
                $data = $result['data'];

                // Handle object response
                if (is_object($data)) {
                    if (isset($data->messages[0]->id)) {
                        $messageId = $data->messages[0]->id;
                    } elseif (isset($data->messages) && is_array($data->messages) && isset($data->messages[0]['id'])) {
                        $messageId = $data->messages[0]['id'];
                    }
                }
                // Handle array response
                elseif (is_array($data)) {
                    if (isset($data['messages'][0]['id'])) {
                        $messageId = $data['messages'][0]['id'];
                    } elseif (isset($data['messages']) && is_array($data['messages']) && ! empty($data['messages'])) {
                        $firstMessage = $data['messages'][0];
                        if (isset($firstMessage['id'])) {
                            $messageId = $firstMessage['id'];
                        }
                    }
                }
                // Handle string response (JSON)
                elseif (is_string($data)) {
                    $decoded = json_decode($data, true);
                    if ($decoded && isset($decoded['messages'][0]['id'])) {
                        $messageId = $decoded['messages'][0]['id'];
                    }
                }
            }

            // Check responseData as fallback
            if (! $messageId && isset($result['responseData'])) {
                $responseData = $result['responseData'];
                if (is_array($responseData) && isset($responseData['messages'][0]['id'])) {
                    $messageId = $responseData['messages'][0]['id'];
                }
            }

            whatsapp_log('Message ID extraction', 'debug', [
                'extracted_id'     => $messageId,
                'data_type'        => gettype($result['data'] ?? null),
                'has_responseData' => isset($result['responseData']),
            ]);
        } catch (\Exception $e) {
            whatsapp_log('Error extracting message ID', 'error', [
                'error'            => $e->getMessage(),
                'result_structure' => array_keys($result ?? []),
            ], $e);
        }

        return $messageId;
    }

    /**
     * Format message content for display in chat UI
     */
    private function formatMessageContent($nodeType, $nodeData, $contactData)
    {
        switch ($nodeType) {
            case 'textMessage':
                $message = $this->replaceFlowVariables($nodeData['message'] ?? '', $contactData);

                return '<p>' . nl2br(e($message)) . '</p>';

            case 'buttonMessage':
                $message = $this->replaceFlowVariables($nodeData['message'] ?? '', $contactData);
                $buttons = $nodeData['buttons'] ?? [];

                $buttonHtml = "<div class='flex flex-col mt-2 space-y-2'>";
                foreach ($buttons as $button) {
                    $buttonHtml .= "<button class='bg-gray-100 text-success-500 px-3 py-2 rounded-lg flex items-center justify-center text-xs space-x-2 w-full
                        dark:bg-gray-800 dark:text-success-400'>" . e($button['text']) . '</button>';
                }
                $buttonHtml .= '</div>';

                return '<p>' . nl2br(e($message)) . '</p>' . $buttonHtml;

            case 'mediaMessage':
                $mediaType = $nodeData['mediaType'] ?? 'image';
                $mediaUrl  = $nodeData['mediaUrl']  ?? '';
                $caption   = $this->replaceFlowVariables($nodeData['caption'] ?? '', $contactData);

                $mediaHtml = '';
                if ($mediaType === 'image') {
                    $mediaHtml = "<a href='{$mediaUrl}' data-lightbox='image-group'>
                        <img src='{$mediaUrl}' class='rounded-lg w-full mb-2'>
                        </a>";
                } elseif ($mediaType === 'video') {
                    $mediaHtml = "<video src='{$mediaUrl}' controls class='rounded-lg w-full'></video>";
                } elseif ($mediaType === 'audio') {
                    $mediaHtml = "<audio controls class='w-64'><source src='{$mediaUrl}' type='audio/mpeg'></audio>";
                } elseif ($mediaType === 'document') {
                    $fileName  = $nodeData['fileName'] ?? basename($mediaUrl);
                    $mediaHtml = "<a href='{$mediaUrl}' target='_blank' class='bg-gray-100 text-success-500 px-3 py-2 rounded-lg flex items-center justify-center text-xs space-x-2 w-full
                       dark:bg-gray-800 dark:text-success-400'>" . e($fileName) . '</a>';
                }

                return $mediaHtml . '<p>' . nl2br(e($caption)) . '</p>';

            case 'listMessage':
                $headerText = $this->replaceFlowVariables($nodeData['headerText'] ?? '', $contactData);
                $bodyText   = $this->replaceFlowVariables($nodeData['bodyText'] ?? '', $contactData);
                $footerText = $this->replaceFlowVariables($nodeData['footerText'] ?? '', $contactData);
                $buttonText = $nodeData['buttonText'] ?? 'View Options';

                $listHtml = '';
                if (! empty($headerText)) {
                    $listHtml .= "<div class='font-medium mb-2'>" . e($headerText) . '</div>';
                }

                $listHtml .= '<p>' . nl2br(e($bodyText)) . '</p>';

                if (! empty($footerText)) {
                    $listHtml .= "<div class='text-gray-500 text-sm mt-1'>" . e($footerText) . '</div>';
                }

                $listHtml .= "<div class='bg-gray-100 text-success-500 px-3 py-2 rounded-lg flex items-center justify-center text-xs space-x-2 w-full mt-2
                   dark:bg-gray-800 dark:text-success-400'>" . e($buttonText) . '</div>';

                return $listHtml;

            case 'locationMessage':
                $name    = $this->replaceFlowVariables($nodeData['name'] ?? '', $contactData);
                $address = $this->replaceFlowVariables($nodeData['address'] ?? '', $contactData);

                return "<div class='bg-gray-100 p-3 rounded-lg'>
                    <div class='font-medium'>" . e($name) . "</div>
                    <div class='text-sm text-gray-500'>" . e($address) . "</div>
                    <div class='text-xs text-info-500 mt-1'>Location: " . e($nodeData['latitude']) . ', ' . e($nodeData['longitude']) . '</div>
                </div>';

            case 'contactMessage':
                $contacts    = $nodeData['contacts'] ?? [];
                $contactHtml = "<div class='space-y-2'>";

                foreach ($contacts as $contact) {
                    $firstName = $this->replaceFlowVariables($contact['firstName'] ?? '', $contactData);
                    $lastName  = $this->replaceFlowVariables($contact['lastName'] ?? '', $contactData);
                    $phone     = $this->replaceFlowVariables($contact['phone'] ?? '', $contactData);

                    $contactHtml .= "<div class='bg-gray-100 p-3 rounded-lg'>
                        <div class='font-medium'>" . e($firstName) . ' ' . e($lastName) . "</div>
                        <div class='text-sm'>" . e($phone) . '</div>';

                    if (! empty($contact['email'])) {
                        $email = $this->replaceFlowVariables($contact['email'], $contactData);
                        $contactHtml .= "<div class='text-sm'>" . e($email) . '</div>';
                    }

                    if (! empty($contact['company'])) {
                        $company = $this->replaceFlowVariables($contact['company'], $contactData);
                        $contactHtml .= "<div class='text-sm text-gray-500'>" . e($company) . '</div>';
                    }

                    $contactHtml .= '</div>';
                }

                $contactHtml .= '</div>';

                return $contactHtml;

            case 'templateMessage':
                // Templates are handled directly by the sendTemplate method
                return "<div class='text-gray-500'>WhatsApp Template Message</div>";

            case 'aiAssistant':
                $prompt = $this->replaceFlowVariables($nodeData['prompt'] ?? '', $contactData);

                return "<p class='text-sm italic text-gray-500'>AI response for: " . e($prompt) . '</p>';

            default:
                return '<p>Unsupported message type</p>';
        }
    }

    /**
     * Extract trigger message from incoming message data
     */
    private function extractTriggerMessage($message)
    {
        // Handle different message types
        if (isset($message['button']['text'])) {
            return $message['button']['text'];
        } elseif (isset($message['text']['body'])) {
            return $message['text']['body'];
        } elseif (! empty($message['interactive'])) {
            if ($message['interactive']['type'] == 'button_reply') {
                return $message['interactive']['button_reply']['id'];
            } elseif ($message['interactive']['type'] == 'list_reply') {
                return $message['interactive']['list_reply']['id'];
            }
        }

        return '';
    }

    /**
     * Check if bot flow should be skipped
     */
    private function shouldSkipBotFlow($interaction, $trigger_msg)
    {
        if (! $interaction) {
            return false;
        }

        // Check if bot is temporarily stopped
        if ($interaction->is_bots_stoped == 1) {
            // Check if restart time has passed
            $restart_after = (int) get_setting('whats-mark.restart_bots_after');
            if ($restart_after > 0 && time() > strtotime($interaction->bot_stoped_time) + ($restart_after * 3600)) {
                // Restart the bot
                Chat::where('id', $interaction->id)->update([
                    'bot_stoped_time' => null,
                    'is_bots_stoped'  => '0',
                ]);

                return false;
            }

            return true;
        }

        // Check if this message should stop the bot
        $stopKeywords = collect(get_setting('whats-mark.stop_bots_keyword'));
        if ($stopKeywords->first(fn ($keyword) => str_contains(strtolower($trigger_msg), strtolower($keyword)))) {
            Chat::where('id', $interaction->id)->update([
                'bot_stoped_time' => date('Y-m-d H:i:s'),
                'is_bots_stoped'  => '1',
            ]);

            return true;
        }

        return false;
    }

    /**
     * Check if flow matches the relation type and trigger
     */
    private function isFlowMatch($triggerNode, $relType, $trigger)
    {
        $nodeData = $triggerNode['data'] ?? [];
        $output   = $nodeData['output']  ?? [];

        // No output rules defined
        if (empty($output)) {
            whatsapp_log('No output rules for trigger', 'debug', [
                'trigger_id' => $triggerNode['id'] ?? 'unknown',
            ]);

            return false;
        }

        // Check each output rule
        foreach ($output as $rule) {
            // Check relation type match
            if (! empty($rule['rel_type']) && $rule['rel_type'] !== $relType) {
                whatsapp_log('Relation type mismatch', 'debug', [
                    'trigger_id'        => $triggerNode['id'] ?? 'unknown',
                    'expected_rel_type' => $rule['rel_type'],
                    'actual_rel_type'   => $relType,
                ]);

                continue;
            }

            // Get reply type from the rule
            $replyType = $rule['reply_type'] ?? 0;

            whatsapp_log('Checking trigger match', 'debug', [
                'trigger_id'       => $triggerNode['id'] ?? 'unknown',
                'reply_type'       => $replyType,
                'rule_trigger'     => $rule['trigger'] ?? '',
                'incoming_trigger' => $trigger,
                'rel_type'         => $relType,
            ]);

            switch ($replyType) {
                case 1: // Exact match
                    if (strtolower($trigger) === strtolower($rule['trigger'] ?? '')) {
                        whatsapp_log('Exact match found', 'info', [
                            'trigger_id'      => $triggerNode['id'] ?? 'unknown',
                            'matched_trigger' => $trigger,
                        ]);

                        return true;
                    }
                    break;

                case 2: // Contains
                    if (! empty($rule['trigger']) && stripos($trigger, $rule['trigger']) !== false) {
                        whatsapp_log('Contains match found', 'info', [
                            'trigger_id'      => $triggerNode['id'] ?? 'unknown',
                            'matched_trigger' => $trigger,
                        ]);

                        return true;
                    }
                    break;

                case 3: // First time
                    // This would need additional logic to check if it's the user's first interaction
                    whatsapp_log('First time trigger check', 'debug', [
                        'trigger_id' => $triggerNode['id'] ?? 'unknown',
                    ]);

                    return true; // Simplified for now

                case 4: // Fallback
                    whatsapp_log('Fallback trigger match', 'info', [
                        'trigger_id' => $triggerNode['id'] ?? 'unknown',
                    ]);

                    return true;
            }
        }

        whatsapp_log('No trigger match found', 'debug', [
            'trigger_id'       => $triggerNode['id'] ?? 'unknown',
            'incoming_trigger' => $trigger,
            'rel_type'         => $relType,
        ]);

        return false;
    }

    /**
     * Evaluate condition based on condition type and value
     */
    private function evaluateCondition($nodeData, $triggerMsg, $contactData)
    {
        $condition      = $nodeData['condition'] ?? [];
        $conditionType  = $condition['type']     ?? 'contains';
        $conditionValue = $this->replaceFlowVariables($condition['value'] ?? '', $contactData);

        // If empty condition, return false
        if (empty($conditionValue)) {
            return false;
        }

        switch ($conditionType) {
            case 'contains':
                return str_contains(strtolower($triggerMsg), strtolower($conditionValue));

            case 'equals':
                return strtolower($triggerMsg) === strtolower($conditionValue);

            case 'startsWith':
                return str_starts_with(strtolower($triggerMsg), strtolower($conditionValue));

            case 'endsWith':
                return str_ends_with(strtolower($triggerMsg), strtolower($conditionValue));

            case 'regex':
                return (bool) preg_match("/{$conditionValue}/i", $triggerMsg);

            default:
                return false;
        }
    }

    private function buildFlowMessageHtml($nodeType, $nodeData, $contactData, $context)
    {
        $output = $nodeData['output'][0] ?? [];

        switch ($nodeType) {
            case 'textMessage':
                $text = $this->replaceFlowVariables($output['reply_text'] ?? '', $contactData);

                return '<p>' . nl2br(decodeWhatsAppSigns(e($text))) . '</p>';

            case 'buttonMessage':
                $text    = $this->replaceFlowVariables($output['reply_text'] ?? '', $contactData);
                $button1 = $output['button1'] ?? '';
                $button2 = $output['button2'] ?? '';
                $button3 = $output['button3'] ?? '';

                $buttonHtml = "<div class='flex flex-col mt-2 space-y-2'>";

                if ($button1) {
                    $buttonHtml .= "<button class='bg-gray-100 text-success-500 px-3 py-2 rounded-lg flex items-center justify-center text-xs space-x-2 w-full dark:bg-gray-800 dark:text-success-400'>" . e($button1) . '</button>';
                }
                if ($button2) {
                    $buttonHtml .= "<button class='bg-gray-100 text-success-500 px-3 py-2 rounded-lg flex items-center justify-center text-xs space-x-2 w-full dark:bg-gray-800 dark:text-success-400'>" . e($button2) . '</button>';
                }
                if ($button3) {
                    $buttonHtml .= "<button class='bg-gray-100 text-success-500 px-3 py-2 rounded-lg flex items-center justify-center text-xs space-x-2 w-full dark:bg-gray-800 dark:text-success-400'>" . e($button3) . '</button>';
                }

                $buttonHtml .= '</div>';

                return '<p>' . nl2br(decodeWhatsAppSigns(e($text))) . '</p>' . $buttonHtml;

            case 'listMessage':
                $text     = $this->replaceFlowVariables($output['reply_text'] ?? '', $contactData);
                $sections = $output['sections'] ?? [];

                $listHtml = '<p>' . nl2br(decodeWhatsAppSigns(e($text))) . '</p>';
                $listHtml .= "<div class='bg-gray-50 rounded-lg p-3 mt-2 dark:bg-gray-800'>";
                $listHtml .= "<div class='text-sm text-gray-600 dark:text-gray-400 mb-2'>📋 " . ($output['buttonText'] ?? 'Select Option') . '</div>';

                foreach ($sections as $section) {
                    $listHtml .= "<div class='mb-2'>";
                    $listHtml .= "<div class='font-semibold text-xs text-gray-700 dark:text-gray-300'>" . e($section['title'] ?? '') . '</div>';

                    foreach ($section['items'] as $item) {
                        $listHtml .= "<div class='bg-white rounded p-2 mt-1 border-l-2 border-success-500 dark:bg-gray-700'>";
                        $listHtml .= "<div class='font-medium text-sm'>" . e($item['title'] ?? '') . '</div>';
                        if (! empty($item['description'])) {
                            $listHtml .= "<div class='text-xs text-gray-500 dark:text-gray-400'>" . e($item['description']) . '</div>';
                        }
                        $listHtml .= '</div>';
                    }
                    $listHtml .= '</div>';
                }

                $listHtml .= '</div>';

                return $listHtml;

            case 'callToAction':
                $header     = $this->replaceFlowVariables($output['bot_header'] ?? '', $contactData);
                $text       = $this->replaceFlowVariables($output['reply_text'] ?? '', $contactData);
                $footer     = $this->replaceFlowVariables($output['bot_footer'] ?? '', $contactData);
                $buttonText = $output['buttonText'] ?? 'Click Here';
                $buttonLink = $output['buttonLink'] ?? '#';

                $ctaHtml = '';
                if ($header) {
                    $ctaHtml .= "<span class='font-semibold mb-1'>" . nl2br(decodeWhatsAppSigns(e($header))) . '</span><br>';
                }
                $ctaHtml .= '<p>' . nl2br(decodeWhatsAppSigns(e($text))) . '</p>';
                if ($footer) {
                    $ctaHtml .= "<span class='text-gray-500 dark:text-gray-400 text-xs mb-2'>" . nl2br(e($footer)) . '</span><br>';
                }

                $ctaHtml .= "<div class='mt-2'>";
                $ctaHtml .= "<a href='" . e($buttonLink) . "' target='_blank' class='bg-primary-600 hover:bg-primary-700 text-white px-3 py-1.5 rounded-md inline-block text-xs font-medium transition'>" . e($buttonText) . '</a>';
                $ctaHtml .= '</div>';

                return $ctaHtml;

            case 'mediaMessage':
                $mediaType = $output['media_type'] ?? 'image';
                $mediaUrl  = $output['media_url']  ?? '';
                $caption   = $this->replaceFlowVariables($output['media_caption'] ?? '', $contactData);

                $mediaHtml = '';

                switch ($mediaType) {
                    case 'image':
                        $mediaHtml = "<a href='" . e($mediaUrl) . "' data-lightbox='image-group'>";
                        $mediaHtml .= "<img src='" . e($mediaUrl) . "' class='rounded-lg w-full mb-2 max-w-sm'>";
                        $mediaHtml .= '</a>';
                        break;
                    case 'video':
                        $mediaHtml = "<video src='" . e($mediaUrl) . "' controls class='rounded-lg w-full max-w-sm'></video>";
                        break;
                    case 'audio':
                        $mediaHtml = "<audio controls class='w-64'><source src='" . e($mediaUrl) . "' type='audio/mpeg'></audio>";
                        break;
                    case 'document':
                        $filename  = $output['media_filename'] ?? basename($mediaUrl);
                        $mediaHtml = "<a href='" . e($mediaUrl) . "' target='_blank' class='bg-gray-100 text-success-500 px-3 py-2 rounded-lg flex items-center justify-center text-xs space-x-2 w-full dark:bg-gray-800 dark:text-success-400'>📄 " . e($filename) . '</a>';
                        break;
                }

                if ($caption) {
                    $mediaHtml .= "<p class='mt-2'>" . nl2br(decodeWhatsAppSigns(e($caption))) . '</p>';
                }

                return $mediaHtml;

            case 'locationMessage':
                $name      = $output['location_name']      ?? 'Location';
                $address   = $output['location_address']   ?? '';
                $latitude  = $output['location_latitude']  ?? '';
                $longitude = $output['location_longitude'] ?? '';

                $locationHtml = "<div class='bg-gray-50 rounded-lg p-3 dark:bg-gray-800'>";
                $locationHtml .= "<div class='flex items-center mb-2'>";
                $locationHtml .= "<span class='text-lg mr-2'>📍</span>";
                $locationHtml .= '<div>';
                $locationHtml .= "<div class='font-semibold'>" . e($name) . '</div>';
                if ($address) {
                    $locationHtml .= "<div class='text-sm text-gray-600 dark:text-gray-400'>" . e($address) . '</div>';
                }
                $locationHtml .= '</div>';
                $locationHtml .= '</div>';

                if ($latitude && $longitude) {
                    $mapUrl = 'https://www.google.com/maps?q=' . urlencode($latitude . ',' . $longitude);
                    $locationHtml .= "<a href='" . $mapUrl . "' target='_blank' class='text-info-500 text-sm hover:underline'>View on Map</a>";
                }

                $locationHtml .= '</div>';

                return $locationHtml;

            case 'contactMessage':
                $contacts = $output['contacts'] ?? [];

                $contactHtml = "<div class='bg-gray-50 rounded-lg p-3 dark:bg-gray-800'>";
                $contactHtml .= "<div class='text-sm text-gray-600 dark:text-gray-400 mb-3 font-medium flex items-center gap-1'>👤 Contact" . (count($contacts) > 1 ? 's' : '') . '</div>';

                foreach ($contacts as $contact) {
                    $contactHtml .= "<div class='bg-white dark:bg-gray-700 rounded-lg p-3 space-y-1.5'>";
                    $contactHtml .= "<div class='text-base font-semibold text-gray-800 dark:text-gray-100'>" . e(($contact['firstName'] ?? '') . ' ' . ($contact['lastName'] ?? '')) . '</div>';
                    if (! empty($contact['phone'])) {
                        $contactHtml .= "<div class='text-sm text-gray-600 dark:text-gray-300'>📞 " . e($contact['phone']) . '</div>';
                    }
                    if (! empty($contact['email'])) {
                        $contactHtml .= "<div class='text-sm text-gray-600 dark:text-gray-300'>✉️ " . e($contact['email']) . '</div>';
                    }
                    if (! empty($contact['company'])) {
                        $contactHtml .= "<div class='text-sm text-gray-600 dark:text-gray-300'>🏢 " . e($contact['company']) . '</div>';
                    }
                    $contactHtml .= '</div>';
                }

                $contactHtml .= '</div>';

                return $contactHtml;

            default:
                return '<p>Flow message: ' . e($nodeType) . '</p>';
        }
    }

    private function extractPlainTextFromFlowMessage($nodeType, $nodeData, $contactData)
    {
        $output = $nodeData['output'][0] ?? [];

        switch ($nodeType) {
            case 'textMessage':
                return $this->replaceFlowVariables($output['reply_text'] ?? '', $contactData);

            case 'buttonMessage':
                $text    = $this->replaceFlowVariables($output['reply_text'] ?? '', $contactData);
                $buttons = array_filter([$output['button1'] ?? '', $output['button2'] ?? '', $output['button3'] ?? '']);

                return $text . (count($buttons) > 0 ? ' [' . count($buttons) . ' buttons]' : '');

            case 'listMessage':
                $text       = $this->replaceFlowVariables($output['reply_text'] ?? '', $contactData);
                $sections   = $output['sections'] ?? [];
                $totalItems = array_sum(array_map(function ($section) {
                    return count($section['items'] ?? []);
                }, $sections));

                return $text . ' [List with ' . $totalItems . ' options]';

            case 'callToAction':
                return $this->replaceFlowVariables($output['reply_text'] ?? '', $contactData) . ' [CTA Button]';

            case 'mediaMessage':
                $caption   = $this->replaceFlowVariables($output['media_caption'] ?? '', $contactData);
                $mediaType = $output['media_type'] ?? 'media';

                return $caption ?: '[' . ucfirst($mediaType) . ' message]';

            case 'locationMessage':
                return $output['location_name'] ?? 'Location shared';

            case 'contactMessage':
                $contacts = $output['contacts'] ?? [];

                return 'Contact' . (count($contacts) > 1 ? 's' : '') . ' shared (' . count($contacts) . ')';

            default:
                return 'Flow message';
        }
    }

    private function extractMediaUrl($nodeType, $nodeData)
    {
        if ($nodeType === 'mediaMessage') {
            $output   = $nodeData['output'][0] ?? [];
            $mediaUrl = $output['media_url']   ?? '';

            // Extract filename from URL for storage
            if ($mediaUrl && strpos($mediaUrl, '/storage/') !== false) {
                return str_replace(url('/storage/'), '', $mediaUrl);
            }
        }

        return null;
    }

    private function mapNodeTypeToMessageType($nodeType)
    {
        $mapping = [
            'textMessage'     => 'text',
            'buttonMessage'   => 'interactive',
            'listMessage'     => 'interactive',
            'callToAction'    => 'interactive',
            'mediaMessage'    => 'text',
            'locationMessage' => 'text',
            'contactMessage'  => 'contacts',
        ];

        return $mapping[$nodeType] ?? 'text';
    }

    private function updateChatLastMessage($chatId, $plainTextMessage)
    {
        try {
            Chat::where('id', $chatId)->update([
                'last_message'  => $plainTextMessage,
                'last_msg_time' => now(),
                'updated_at'    => now(),
            ]);

            whatsapp_log('Chat last message updated', 'debug', [
                'chat_id'              => $chatId,
                'last_message_preview' => substr($plainTextMessage, 0, 50),
            ]);
        } catch (\Exception $e) {
            whatsapp_log('Error updating chat last message', 'error', [
                'chat_id' => $chatId,
                'error'   => $e->getMessage(),
            ], $e);
        }
    }

    private function triggerChatNotification($chatId, $messageDbId)
    {
        try {
            // Only trigger if Pusher is configured (same check as existing code)
            if (
                ! empty(get_setting('pusher.app_key')) && ! empty(get_setting('pusher.app_secret')) && ! empty(get_setting('pusher.app_id')) && ! empty(get_setting('pusher.cluster'))
            ) {

                $pusherService = new PusherService;
                $pusherService->trigger('whatsmark-chat-channel', 'whatsmark-chat-event', [
                    'chat' => ChatController::newChatMessage($chatId, $messageDbId),
                ]);

                whatsapp_log('Chat notification triggered', 'debug', [
                    'chat_id'       => $chatId,
                    'message_db_id' => $messageDbId,
                ]);
            }
        } catch (\Exception $e) {
            whatsapp_log('Error triggering chat notification', 'error', [
                'chat_id'       => $chatId,
                'message_db_id' => $messageDbId,
                'error'         => $e->getMessage(),
            ], $e);
        }
    }
}
