<?php

namespace App\Services;

use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Log;
use Nwidart\Modules\Facades\Module;
use ZipArchive;

class ModuleUploader
{
    protected string $modulesPath;

    protected string $tempPath;

    protected ModuleValidator $validator;

    public function __construct(ModuleValidator $validator)
    {
        $this->modulesPath = base_path('Modules');
        $this->tempPath    = storage_path('app/temp');
        $this->validator   = $validator;

        $this->ensureDirectoriesExist();
    }

    /**
     * Process module upload, validate and install it
     *
     * @param  UploadedFile $file The uploaded module ZIP file
     * @return array        Result with status and message
     */
    public function upload(UploadedFile $file): array
    {
        try {
            // Stage 1: Process the uploaded ZIP file
            $result = $this->processUploadedFile($file);
            if (! $result['status']) {
                return $result;
            }

            $extractPath = $result['extract_path'];
            $zipPath     = $result['zip_path'];

            // Stage 2: Find and validate module.json
            $moduleJsonPath = $this->findModuleJson($extractPath);
            if (! $moduleJsonPath) {
                $this->cleanup($extractPath, $zipPath);

                return [
                    'status'  => false,
                    'message' => 'module.json not found in the uploaded package.',
                ];
            }

            // Stage 3: Validate module
            if (! $this->validator->validate($moduleJsonPath, dirname($moduleJsonPath))) {
                $this->cleanup($extractPath, $zipPath);

                return [
                    'status'  => false,
                    'message' => implode("\n", $this->validator->getErrors()),
                ];
            }

            // Stage 4: Install module
            $moduleJson = json_decode(File::get($moduleJsonPath), true);
            $moduleName = $moduleJson['name'];
            $moduleDir  = dirname($moduleJsonPath);

            $result = $this->installModule($moduleName, $moduleDir);

            // Cleanup regardless of success or failure
            $this->cleanup($extractPath, $zipPath);

            return $result;

        } catch (\Exception $e) {
            Log::error('Module upload failed: ' . $e->getMessage(), [
                'file'  => $e->getFile(),
                'line'  => $e->getLine(),
                'trace' => $e->getTraceAsString(),
            ]);

            // Cleanup on error
            if (isset($extractPath) && File::exists($extractPath)) {
                File::deleteDirectory($extractPath);
            }
            if (isset($zipPath) && File::exists($zipPath)) {
                File::delete($zipPath);
            }

            return [
                'status'  => false,
                'message' => 'Error processing module: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Process uploaded ZIP file
     */
    protected function processUploadedFile(UploadedFile $file): array
    {
        $zipFileName = time() . '_' . $file->getClientOriginalName();
        $zipPath     = $this->tempPath . '/' . $zipFileName;

        // Move uploaded file to temp directory
        $file->move($this->tempPath, $zipFileName);

        // Open zip archive
        $zip = new ZipArchive;
        if ($zip->open($zipPath) !== true) {
            File::delete($zipPath);

            return [
                'status'  => false,
                'message' => 'Unable to open the zip file.',
            ];
        }

        // Create a temporary extraction directory
        $extractPath = $this->tempPath . '/extract_' . time();
        File::makeDirectory($extractPath, 0755, true);

        // Extract the zip file
        $zip->extractTo($extractPath);
        $zip->close();

        return [
            'status'       => true,
            'zip_path'     => $zipPath,
            'extract_path' => $extractPath,
        ];
    }

    /**
     * Find module.json in the extracted module
     */
    protected function findModuleJson(string $path): ?string
    {
        // Direct check
        if (File::exists($path . '/module.json')) {
            return $path . '/module.json';
        }

        // Check first level subdirectories
        foreach (File::directories($path) as $dir) {
            if (File::exists($dir . '/module.json')) {
                return $dir . '/module.json';
            }

            // Check second level subdirectories
            foreach (File::directories($dir) as $subdir) {
                if (File::exists($subdir . '/module.json')) {
                    return $subdir . '/module.json';
                }
            }
        }

        return null;
    }

    /**
     * Install the module by moving files and registering it
     */
    protected function installModule(string $moduleName, string $moduleDir): array
    {
        $finalModulePath = $this->modulesPath . '/' . $moduleName;

        // Check if trying to replace a core module
        if ($this->validator->isCoreModule($moduleName) && File::exists($finalModulePath)) {
            return [
                'status'  => false,
                'message' => 'Cannot replace a core module.',
            ];
        }

        // Backup existing module if needed
        if (File::exists($finalModulePath)) {
            $backupPath = $this->createBackup($moduleName, $finalModulePath);
            if (! $backupPath) {
                return [
                    'status'  => false,
                    'message' => 'Failed to create backup of existing module.',
                ];
            }
        }

        // Move module files to Modules directory
        if (File::exists($finalModulePath)) {
            File::deleteDirectory($finalModulePath);
        }

        // Create module directory and copy files
        File::makeDirectory($finalModulePath, 0755, true);
        File::copyDirectory($moduleDir, $finalModulePath);

        // Update modules_statuses.json
        $this->updateModuleStatus($moduleName);

        // Run post-installation tasks
        return $this->runPostInstallation($moduleName, $finalModulePath);
    }

    /**
     * Create backup of existing module
     */
    protected function createBackup(string $moduleName, string $modulePath): ?string
    {
        try {
            $backupDir = storage_path('app/module_backups');
            if (! File::exists($backupDir)) {
                File::makeDirectory($backupDir, 0755, true);
            }

            $backupPath = $backupDir . '/' . $moduleName . '_' . date('Y-m-d-His');
            File::copyDirectory($modulePath, $backupPath);

            return $backupPath;
        } catch (\Exception $e) {
            Log::error('Failed to create module backup: ' . $e->getMessage());

            return null;
        }
    }

    /**
     * Update the module status in modules_statuses.json
     */
    protected function updateModuleStatus(string $moduleName): void
    {
        $modulesJsonPath = base_path('modules_statuses.json');
        $modulesJson     = File::exists($modulesJsonPath)
            ? json_decode(File::get($modulesJsonPath), true)
            : [];

        $modulesJson[$moduleName] = false; // Initially set as disabled
        File::put($modulesJsonPath, json_encode($modulesJson, JSON_PRETTY_PRINT));
    }

    /**
     * Run post-installation tasks
     */
    protected function runPostInstallation(string $moduleName, string $modulePath): array
    {
        try {
            // Run migrations if they exist
            if (File::exists($modulePath . '/Database/Migrations')) {
                Artisan::call('module:migrate', ['module' => $moduleName]);
            }

            // Enable the module
            $module = Module::find($moduleName);
            if ($module) {
                $module->enable();
            }

            // Run seeders if they exist
            if (File::exists($modulePath . '/Database/Seeders')) {
                Artisan::call('module:seed', ['module' => $moduleName]);
            }

            // Publish module assets
            $this->publishModuleAssets($moduleName);

            // Clear caches
            $this->clearCaches();

            return [
                'status'  => true,
                'message' => "Module '$moduleName' has been uploaded and installed successfully!",
            ];
        } catch (\Exception $e) {
            Log::error('Post-installation failed: ' . $e->getMessage());

            return [
                'status'  => false,
                'message' => 'Module was uploaded but installation failed: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Publish module assets
     */
    protected function publishModuleAssets(string $moduleName): void
    {
        $publishers = [
            'config'       => 'module:publish-config',
            'migrations'   => 'module:publish-migration',
            'assets'       => 'module:publish-asset',
            'views'        => 'module:publish-views',
            'translations' => 'module:publish-translation',
        ];

        foreach ($publishers as $type => $command) {
            try {
                Artisan::call($command, ['module' => $moduleName]);
            } catch (\Exception $e) {
                Log::warning("Failed to publish $type for module $moduleName: " . $e->getMessage());
                // Continue with other publishers even if one fails
            }
        }
    }

    /**
     * Clear application caches
     */
    protected function clearCaches(): void
    {
        $caches = ['cache:clear', 'config:clear', 'route:clear', 'view:clear'];

        foreach ($caches as $cache) {
            try {
                Artisan::call($cache);
            } catch (\Exception $e) {
                Log::warning("Failed to clear cache ($cache): " . $e->getMessage());
            }
        }
    }

    /**
     * Cleanup temporary files
     */
    protected function cleanup(string $extractPath, string $zipPath): void
    {
        if (File::exists($extractPath)) {
            File::deleteDirectory($extractPath);
        }

        if (File::exists($zipPath)) {
            File::delete($zipPath);
        }
    }

    /**
     * Ensure required directories exist
     */
    protected function ensureDirectoriesExist(): void
    {
        foreach ([$this->tempPath, $this->modulesPath] as $dir) {
            if (! File::exists($dir)) {
                File::makeDirectory($dir, 0755, true);
            }
        }
    }
}
