<?php

namespace App\Services;

use Illuminate\Support\Facades\File;
use Nwidart\Modules\Facades\Module;

class ModuleValidator
{
    protected array $errors = [];

    protected array $coreModules;

    public function __construct()
    {
        $this->coreModules = config('modules.core_modules', []);
    }

    /**
     * Validate a module.json file
     */
    public function validate(string $moduleJsonPath, string $moduleDir): bool
    {
        if (! File::exists($moduleJsonPath)) {
            $this->errors[] = 'module.json file not found';

            return false;
        }

        try {
            $content    = File::get($moduleJsonPath);
            $moduleJson = json_decode($content, true);

            if (json_last_error() !== JSON_ERROR_NONE) {
                $this->errors[] = 'Invalid JSON format in module.json';

                return false;
            }

            return $this->validateRequiredFields($moduleJson) && $this->validateVersion($moduleJson) && $this->validateProviders($moduleJson) && $this->validateRequiredFiles($moduleJson, $moduleDir) && $this->validateDependencies($moduleJson) && $this->validateSystemCompatibility($moduleJson);
        } catch (\Exception $e) {
            $this->errors[] = 'Error validating module.json: ' . $e->getMessage();

            return false;
        }
    }

    /**
     * Check that all required fields are present in module.json
     */
    protected function validateRequiredFields(array $moduleJson): bool
    {
        $requiredFields = config('modules.required_fields', ['name', 'alias', 'description', 'version', 'providers']);
        $missingFields  = [];

        foreach ($requiredFields as $field) {
            if (! isset($moduleJson[$field]) || (is_string($moduleJson[$field]) && trim($moduleJson[$field]) === '')) {
                $missingFields[] = $field;
            }
        }

        if (! empty($missingFields)) {
            $this->errors[] = 'Missing required fields in module.json: ' . implode(', ', $missingFields);

            return false;
        }

        return true;
    }

    /**
     * Validate version format
     */
    protected function validateVersion(array $moduleJson): bool
    {
        if (! isset($moduleJson['version'])) {
            return false;
        }

        $pattern = config('modules.version_pattern', '/^\d+\.\d+\.\d+$/');

        if (! preg_match($pattern, $moduleJson['version'])) {
            $this->errors[] = 'Invalid version format in module.json. Must follow semantic versioning (e.g., 1.0.0)';

            return false;
        }

        return true;
    }

    /**
     * Validate module providers
     */
    protected function validateProviders(array $moduleJson): bool
    {
        if (! isset($moduleJson['providers']) || ! is_array($moduleJson['providers']) || empty($moduleJson['providers'])) {
            $this->errors[] = 'Module must have at least one service provider';

            return false;
        }

        foreach ($moduleJson['providers'] as $provider) {
            if (! is_string($provider)) {
                $this->errors[] = 'Invalid provider format in module.json';

                return false;
            }
        }

        return true;
    }

    /**
     * Validate that required files exist in the module
     */
    protected function validateRequiredFiles(array $moduleJson, string $moduleDir): bool
    {
        // Check if files array exists in module.json
        if (! isset($moduleJson['files']) || ! is_array($moduleJson['files'])) {
            return true; // No files specified, validation passes
        }

        $missingFiles = [];

        foreach ($moduleJson['files'] as $file) {
            $filePath = $moduleDir . '/' . $file;

            // Normalize directory separators for different OS
            $filePath = str_replace(['/', '\\'], DIRECTORY_SEPARATOR, $filePath);

            if (! File::exists($filePath)) {
                $missingFiles[] = $file;
            }
        }

        if (! empty($missingFiles)) {
            $this->errors[] = 'Required files missing in module: ' . implode(', ', $missingFiles);

            return false;
        }

        return true;
    }

    /**
     * Validate module dependencies
     */
    protected function validateDependencies(array $moduleJson): bool
    {
        if (! isset($moduleJson['requires_modules']) || ! is_array($moduleJson['requires_modules'])) {
            return true; // No module dependencies specified
        }

        $missingDependencies = [];

        foreach ($moduleJson['requires_modules'] as $dependencyName => $version) {
            $dependencyModule = Module::find($dependencyName);

            if (! $dependencyModule || ! $dependencyModule->isEnabled()) {
                $missingDependencies[] = $dependencyName . ' (v' . $version . ')';
            } else {
                // Check version requirement if specified
                $dependencyVersion = $dependencyModule->get('version');
                if ($dependencyVersion && version_compare($dependencyVersion, $version, '<')) {
                    $missingDependencies[] = $dependencyName . ' (requires v' . $version . ', found v' . $dependencyVersion . ')';
                }
            }
        }

        if (! empty($missingDependencies)) {
            $this->errors[] = 'Missing dependencies: ' . implode(', ', $missingDependencies);

            return false;
        }

        return true;
    }

    /**
     * Check system compatibility
     */
    protected function validateSystemCompatibility(array $moduleJson): bool
    {
        // Check PHP version
        if (isset($moduleJson['requires']['php'])) {
            $requiredPhp = $moduleJson['requires']['php'];
            if (substr($requiredPhp, 0, 1) === '^') {
                $requiredPhp = substr($requiredPhp, 1);
            }

            if (version_compare(PHP_VERSION, $requiredPhp, '<')) {
                $this->errors[] = 'This module requires PHP v' . $requiredPhp . ' or higher. Current version: ' . PHP_VERSION;

                return false;
            }
        }

        // Check Laravel version
        if (isset($moduleJson['requires']['laravel/framework'])) {
            $requiredLaravel = $moduleJson['requires']['laravel/framework'];
            if (substr($requiredLaravel, 0, 1) === '^') {
                $requiredLaravel = substr($requiredLaravel, 1);
            }

            $currentLaravel = app()->version();
            if (version_compare($currentLaravel, $requiredLaravel, '<')) {
                $this->errors[] = 'This module requires Laravel v' . $requiredLaravel . ' or higher. Current version: ' . $currentLaravel;

                return false;
            }
        }

        return true;
    }

    /**
     * Check if a module is a core module
     */
    public function isCoreModule(string $moduleName): bool
    {
        return in_array($moduleName, $this->coreModules);
    }

    /**
     * Get validation errors
     */
    public function getErrors(): array
    {
        return $this->errors;
    }
}
