<?php

namespace App\Traits;

use App\DTOs\WhatsAppMessage;
use App\Exceptions\WhatsAppException;
use App\Jobs\SendWhatsAppMessage;
use App\Models\ChatMessage;
use App\Models\WhatsappTemplate;
use App\Models\WmActivityLog;
use App\Services\PusherService;
use Endroid\QrCode\Color\Color;
use Endroid\QrCode\Encoding\Encoding;
use Endroid\QrCode\ErrorCorrectionLevel;
use Endroid\QrCode\Label\Label;
use Endroid\QrCode\Logo\Logo;
use Endroid\QrCode\QrCode;
use Endroid\QrCode\RoundBlockSizeMode;
use Endroid\QrCode\Writer\PngWriter;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Queue;
use Netflie\WhatsAppCloudApi\Message\ButtonReply\Button;
use Netflie\WhatsAppCloudApi\Message\ButtonReply\ButtonAction;
use Netflie\WhatsAppCloudApi\Message\CtaUrl\TitleHeader;
use Netflie\WhatsAppCloudApi\Message\Media\LinkID;
use Netflie\WhatsAppCloudApi\Message\Template\Component;
use Netflie\WhatsAppCloudApi\Response\ResponseException;
use Netflie\WhatsAppCloudApi\WhatsAppCloudApi;
use Storage;
use Throwable;

trait WhatsApp
{
    protected static string $facebookAPI = 'https://graph.facebook.com/';

    protected static array $extensionMap = [
        'image/jpeg'                                                                => 'jpg',
        'image/png'                                                                 => 'png',
        'audio/mp3'                                                                 => 'mp3',
        'video/mp4'                                                                 => 'mp4',
        'audio/aac'                                                                 => 'aac',
        'audio/amr'                                                                 => 'amr',
        'audio/ogg'                                                                 => 'ogg',
        'audio/mp4'                                                                 => 'mp4',
        'text/plain'                                                                => 'txt',
        'application/pdf'                                                           => 'pdf',
        'application/vnd.ms-powerpoint'                                             => 'ppt',
        'application/msword'                                                        => 'doc',
        'application/vnd.ms-excel'                                                  => 'xls',
        'application/vnd.openxmlformats-officedocument.wordprocessingml.document'   => 'docx',
        'application/vnd.openxmlformats-officedocument.presentationml.presentation' => 'pptx',
        'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'         => 'xlsx',
        'video/3gp'                                                                 => '3gp',
        'image/webp'                                                                => 'webp',
    ];

    protected static function getApiVersion(): string
    {
        return get_setting('whatsapp.api_version', 'v21.0');
    }

    protected static function getBaseUrl(): string
    {
        return self::$facebookAPI . self::getApiVersion() . '/';
    }

    protected function handleApiError(Throwable $e, string $operation, array $context = []): array
    {
        $errorContext = array_merge([
            'operation'  => $operation,
            'trace'      => $e->getTraceAsString(),
            'account_id' => $this->getAccountID(),
            'phone_id'   => $this->getPhoneID(),
        ], $context);

        whatsapp_log("[WhatsApp {$operation} Error] " . $e->getMessage(), 'error', $errorContext, $e);

        return [
            'status'  => false,
            'message' => config('app.debug')
                ? $e->getMessage()
                : __('whatsapp.errors.' . $operation, ['default' => 'An error occurred during ' . $operation]),
        ];
    }

    private function getToken(): string
    {
        return get_setting('whatsapp.wm_access_token');
    }

    private function getAccountID(): string
    {
        return get_setting('whatsapp.wm_business_account_id');
    }

    private function getPhoneID(): string
    {
        return get_setting('whatsapp.wm_default_phone_number_id');
    }

    private function getFBAppID(): string
    {
        return get_setting('whatsapp.wm_fb_app_id');
    }

    private function getFBAppSecret(): string
    {
        return get_setting('whatsapp.wm_fb_app_secret');
    }

    /**
     * Load WhatsApp Cloud API configuration
     *
     * @param  string|null      $fromNumber Optional phone number to use as the sender
     * @return WhatsAppCloudApi Instance of the WhatsAppCloudApi class
     */
    public function loadConfig($fromNumber = null)
    {
        return new WhatsAppCloudApi([
            'from_phone_number_id' => (! empty($fromNumber)) ? $fromNumber : $this->getPhoneID(),
            'access_token'         => $this->getToken(),
        ]);
    }

    public function getPhoneNumbers(): array
    {
        try {

            $response = Http::get(self::getBaseUrl() . "{$this->getAccountID()}/phone_numbers", [
                'access_token' => $this->getToken(),
            ]);

            if ($response->failed()) {
                throw new WhatsAppException($response->json('error.message'));
            }

            return ['status' => true, 'data' => $response->json('data')];
        } catch (Throwable $e) {
            return $this->handleApiError($e, 'get_phone_numbers');
        }
    }

    public function loadTemplatesFromWhatsApp(): array
    {
        try {
            $accountId   = $this->getAccountID();
            $accessToken = $this->getToken();

            $templates = [];
            $url       = self::getBaseUrl() . "{$accountId}/message_templates?limit=100&access_token={$accessToken}";

            // Fetch all templates using pagination
            do {
                $response = Http::get($url);

                if ($response->failed()) {
                    throw new WhatsAppException($response->json('error.message'));
                }

                $data = $response->json('data');
                if (! $data) {
                    throw new WhatsAppException('Message templates not found.');
                }

                $templates = array_merge($templates, $data);

                $url = $response->json('paging.next') ?? null;
            } while ($url);

            // Get existing template IDs from database to track what should be deleted
            $existingTemplateIds = WhatsappTemplate::pluck('template_id')->toArray();
            $apiTemplateIds      = [];

            foreach ($templates as $templateData) {
                $apiTemplateIds[] = $templateData['id'];
                $template         = [
                    'template_name' => $templateData['name'],
                    'language'      => $templateData['language'],
                    'status'        => $templateData['status'],
                    'category'      => $templateData['category'],
                    'id'            => $templateData['id'],
                ];

                $components        = [];
                $headerText        = $bodyText = $footerText = $buttonsData = [];
                $headerParamsCount = $bodyParamsCount = $footerParamsCount = 0;

                foreach ($templateData['components'] as $component) {
                    if (($component['type'] ?? '') === 'HEADER') {
                        $components['TYPE'] = $component['format'] ?? null;
                        if (isset($component['text'])) {
                            $headerText           = $component['text'];
                            $headerParamsCount    = preg_match_all('/{{(.*?)}}/i', $headerText, $matches);
                            $components['HEADER'] = $headerText;
                        }
                    }
                    if (($component['type'] ?? '') === 'BODY' && isset($component['text'])) {
                        $bodyText           = $component['text'];
                        $bodyParamsCount    = preg_match_all('/{{(.*?)}}/i', $bodyText, $matches);
                        $components['BODY'] = $bodyText;
                    }
                    if (($component['type'] ?? '') === 'FOOTER' && isset($component['text'])) {
                        $footerText           = $component['text'];
                        $footerParamsCount    = preg_match_all('/{{(.*?)}}/i', $footerText, $matches);
                        $components['FOOTER'] = $footerText;
                    }
                    if (($component['type'] ?? '') === 'BUTTONS') {
                        $components['BUTTONS'] = isset($component['buttons']) ? json_encode($component['buttons']) : null;
                    }
                }

                $template['header_data_text']    = $components['HEADER']  ?? null;
                $template['header_data_format']  = $components['TYPE']    ?? null;
                $template['body_data']           = $components['BODY']    ?? null;
                $template['footer_data']         = $components['FOOTER']  ?? null;
                $template['buttons_data']        = $components['BUTTONS'] ?? null;
                $template['header_params_count'] = $headerParamsCount;
                $template['body_params_count']   = $bodyParamsCount;
                $template['footer_params_count'] = $footerParamsCount;

                WhatsappTemplate::updateOrCreate(
                    ['template_id' => $templateData['id']],
                    $template
                );
            }

            // Delete templates that exist in DB but not in API
            $templatesForDeletion = array_diff($existingTemplateIds, $apiTemplateIds);
            if (! empty($templatesForDeletion)) {
                $deletedCount = WhatsappTemplate::whereIn('template_id', $templatesForDeletion)->delete();
                whatsapp_log('Deleted templates during sync', 'info', [
                    'deleted_count' => $deletedCount,
                    'template_ids'  => $templatesForDeletion,
                ]);
            }

            return [
                'status' => true,
                'data'   => $templates,
                'synced' => [
                    'updated_or_created' => count($apiTemplateIds),
                    'deleted'            => count($templatesForDeletion),
                ],
                'message' => 'Templates synced successfully',
            ];
        } catch (Throwable $e) {
            return $this->handleApiError($e, 'load_templates');
        }
    }

    public function subscribeWebhook()
    {
        $accessToken = $this->getToken();
        $accountId   = $this->getAccountID();
        $url         = self::$facebookAPI . "/$accountId/subscribed_apps?access_token=" . $accessToken;

        try {
            $response = Http::post($url);

            $data = $response->json();

            if (isset($data['error'])) {
                return [
                    'status'  => false,
                    'message' => $data['error']['message'],
                ];
            }

            return [
                'status' => true,
                'data'   => $data,
            ];
        } catch (\Throwable $th) {
            whatsapp_log('Failed to subscribe webhook: ' . $th->getMessage(), 'error', [
                'url'        => $url,
                'account_id' => $accountId,
            ], $th);

            return [
                'status'  => false,
                'message' => 'Something went wrong: ' . $th->getMessage(),
            ];
        }
    }

    public function queueMessage(WhatsAppMessage $message): array
    {
        try {
            dispatch(new SendWhatsAppMessage($message))
                ->onQueue(json_decode(get_setting('whatsapp.queue'), true)['name']);

            return ['status' => true, 'message' => 'Message queued successfully'];
        } catch (Throwable $e) {
            return $this->handleApiError($e, 'queue_message', [
                'message' => $message->toArray(),
            ]);
        }
    }

    public function debugToken(): array
    {
        try {
            $accessToken = $this->getToken();
            $response    = Http::get(self::getBaseUrl() . 'debug_token', [
                'input_token'  => $accessToken,
                'access_token' => $accessToken,
            ]);

            if ($response->failed()) {
                throw new WhatsAppException($response->json('error.message'));
            }

            return ['status' => true, 'data' => $response->json('data')];
        } catch (Throwable $e) {
            return $this->handleApiError($e, 'debug_token');
        }
    }

    public function getProfile(): array
    {
        try {
            $response = Http::get(self::getBaseUrl() . $this->getPhoneID() . '/whatsapp_business_profile', [
                'fields'       => 'profile_picture_url',
                'access_token' => $this->getToken(),
            ]);

            if ($response->failed()) {
                throw new WhatsAppException($response->json('error.message'));
            }

            return ['status' => true, 'data' => $response->json('data')];
        } catch (Throwable $e) {
            return $this->handleApiError($e, 'get_profile');
        }
    }

    public function getHealthStatus(): array
    {
        try {
            $response = Http::get(self::getBaseUrl() . $this->getAccountID(), [
                'fields'       => 'health_status',
                'access_token' => $this->getToken(),
            ]);

            if ($response->failed()) {
                throw new WhatsAppException($response->json('error.message'));
            }

            return ['status' => true, 'data' => $response->json()];
        } catch (Throwable $e) {
            return $this->handleApiError($e, 'health_status');
        }
    }

    public function getMessageLimit(): array
    {
        $startTime = strtotime(date('Y-m-d 00:00:00'));
        $endTime   = strtotime(date('Y-m-d 23:59:59'));
        try {

            $response = Http::get(self::getBaseUrl() . $this->getAccountID(), [
                'fields'       => "id,name,analytics.start({$startTime}).end({$endTime}).granularity(DAY)",
                'access_token' => $this->getToken(),
            ]);

            if ($response->failed()) {
                throw new WhatsAppException($response->json('error.message'));
            }

            return ['status' => true, 'data' => $response->json()];
        } catch (Throwable $e) {
            return $this->handleApiError($e, 'health_status');
        }
    }

    public function generateUrlQR(string $url, ?string $logo = null): bool
    {
        try {
            $writer = new PngWriter;

            $qrCode = new QrCode(
                data: $url,
                encoding: new Encoding('UTF-8'),
                errorCorrectionLevel: ErrorCorrectionLevel::Low,
                size: 300,
                margin: 10,
                roundBlockSizeMode: RoundBlockSizeMode::Margin,
                foregroundColor: new Color(0, 0, 0),
                backgroundColor: new Color(255, 255, 255)
            );

            if ($logo) {
                $logo = new Logo(
                    path: public_path('img/whatsapp.png'),
                    resizeToWidth: 50,
                    punchoutBackground: true
                );
            }

            // Create generic label
            $label = new Label(
                text: '',
                textColor: new Color(255, 0, 0)
            );

            // Generate the QR code
            $result = $writer->write($qrCode, $logo, $label);

            create_storage_link();

            // Define the path to save the file
            $filePath = storage_path('app/public/images/qrcode.png');

            // Ensure the directory exists
            if (! file_exists(dirname($filePath))) {
                mkdir(dirname($filePath), 0755, true);
            }

            // Save the QR code to the file
            $result->saveToFile($filePath);

            return true;
        } catch (Throwable $e) {
            whatsapp_log('Error generating QR code: ' . $e->getMessage(), 'error', [
                'url'  => $url,
                'logo' => $logo,
            ], $e);

            return false;
        }
    }

    public function connectWebhook()
    {
        $appId     = $this->getFBAppID();
        $appSecret = $this->getFBAppSecret();

        try {
            $url = self::$facebookAPI . $appId . '/subscriptions?access_token=' . $appId . '|' . $appSecret;

            $response = Http::post($url, [
                'object'       => 'whatsapp_business_account',
                'fields'       => 'messages,message_template_quality_update,message_template_status_update,account_update',
                'callback_url' => route('whatsapp.webhook'),
                'verify_token' => get_setting('whatsapp.webhook_verify_token'),
            ]);

            $data = $response->json();

            if (isset($data['error'])) {
                return [
                    'status'  => false,
                    'message' => $data['error']['message'],
                ];
            }

            return [
                'status' => true,
                'data'   => $data,
            ];
        } catch (\Throwable $th) {
            whatsapp_log('Error connecting webhook: ' . $th->getMessage(), 'error', [], $th);

            return [
                'status'  => false,
                'message' => 'Something went wrong: ' . $th->getMessage(),
            ];
        }
    }

    public function disconnectWebhook()
    {
        $appId     = $this->getFBAppID();
        $appSecret = $this->getFBAppSecret();

        $url = self::$facebookAPI . $appId . '/subscriptions?access_token=' . $appId . '|' . $appSecret;

        try {
            $response = Http::delete($url, [], [
                'object' => 'whatsapp_business_account',
                'fields' => 'messages,message_template_quality_update,message_template_status_update,account_update',
            ]);

            $data = $response->json();

            if (isset($data['error'])) {
                return [
                    'status'  => false,
                    'message' => $data['error']['message'],
                ];
            }

            return [
                'status' => true,
                'data'   => $data,
            ];
        } catch (\Throwable $th) {
            whatsapp_log('Error disconnecting webhook: ' . $th->getMessage(), 'error', [], $th);

            return [
                'status'  => false,
                'message' => 'Something went wrong: ' . $th->getMessage(),
            ];
        }
    }

    public function sendTestMessages($number)
    {
        $whatsapp_cloud_api = $this->loadConfig();

        try {
            $result       = $whatsapp_cloud_api->sendTemplate($number, 'hello_world', 'en_US');
            $status       = true;
            $message      = t('whatsapp_message_sent_successfully');
            $data         = json_decode($result->body());
            $responseCode = $result->httpStatusCode();
        } catch (\Netflie\WhatsAppCloudApi\Response\ResponseException $th) {
            $status       = false;
            $message      = $th->responseData()['error']['message'] ?? $th->rawResponse() ?? json_decode($th->getMessage());
            $responseCode = $th->httpStatusCode();

            whatsapp_log('Error sending test message: ' . $message, 'error', [
                'number'        => $number,
                'response_code' => $responseCode,
            ], $th);
        }

        return ['status' => $status, 'message' => $message ?? ''];
    }

    public function checkServiceHealth(): array
    {
        try {
            $healthData = [
                'api_status'      => $this->getHealthStatus(),
                'queue_size'      => Queue::size(json_decode(get_setting('whatsapp.queue'), true)['name']),
                'daily_api_calls' => Cache::get('whatsapp_api_calls_' . now()->format('Y-m-d')),
                'token_status'    => $this->debugToken(),
                'profile_status'  => $this->getProfile(),
            ];

            whatsapp_log(
                'WhatsApp service health check',
                'info',
                $healthData
            );

            return ['status' => true, 'data' => $healthData];
        } catch (Throwable $e) {
            return $this->handleApiError($e, 'health_check');
        }
    }

    protected function getExtensionForType(string $mimeType): ?string
    {
        return self::$extensionMap[$mimeType] ?? null;
    }

    /**
     * Send a template message using the WhatsApp Cloud API
     *
     * @param  string      $to            Recipient phone number
     * @param  array       $template_data Data for the template message
     * @param  string      $type          Type of the message, default is 'campaign'
     * @param  string|null $fromNumber    Optional sender phone number
     * @return array       Response containing status, log data, and any response data or error message
     */
    public function sendTemplate($to, $template_data, $type = 'campaign', $fromNumber = null)
    {
        $rel_type    = $template_data['rel_type'];
        $header_data = [];
        if ($template_data['header_data_format'] == 'TEXT') {
            $header_data = parseText($rel_type, 'header', $template_data, 'array');
        }
        $body_data    = parseText($rel_type, 'body', $template_data, 'array');
        $buttons_data = parseText($rel_type, 'footer', $template_data, 'array');

        $component_header = $component_body = $component_buttons = [];
        $file_link        = asset('storage/' . $template_data['filename']);

        $template_buttons_data = json_decode($template_data['buttons_data']);
        $is_flow               = false;
        if (! empty($template_buttons_data)) {
            $button_types = array_column($template_buttons_data, 'type');
            $is_flow      = in_array('FLOW', $button_types);
        }

        $component_header  = $this->buildHeaderComponent($template_data, $file_link, $header_data);
        $component_body    = $this->buildTextComponent($body_data);
        $component_buttons = $this->buildTextComponent($buttons_data);

        if ($is_flow) {
            $buttons             = json_decode($template_data['buttons_data']);
            $flow_id             = reset($buttons)->flow_id;
            $component_buttons[] = [
                'type'       => 'button',
                'sub_type'   => 'FLOW',
                'index'      => 0,
                'parameters' => [
                    [
                        'type'   => 'action',
                        'action' => [
                            'flow_token' => json_encode(['flow_id' => $flow_id, 'rel_data' => $template_data['flow_action_data'] ?? []]),
                        ],
                    ],
                ],
            ];
        }

        $whatsapp_cloud_api = $this->loadConfig($fromNumber);

        try {
            $components   = new Component($component_header, $component_body, $component_buttons);
            $result       = $whatsapp_cloud_api->sendTemplate($to, $template_data['template_name'], $template_data['language'], $components);
            $status       = true;
            $data         = json_decode($result->body());
            $responseCode = $result->httpStatusCode();
            $responseData = json_encode($result->decodedBody());
            $rawData      = json_encode($result->request()->body());
        } catch (\Netflie\WhatsAppCloudApi\Response\ResponseException $th) {
            $status       = false;
            $message      = $th->responseData()['error']['message'] ?? $th->rawResponse() ?? json_decode($th->getMessage());
            $responseCode = $th->httpStatusCode();
            $responseData = json_encode($message);
            $rawData      = json_encode([]);

            whatsapp_log('Error sending template: ' . $message, 'error', [
                'to'            => $to,
                'template_name' => $template_data['template_name'],
                'language'      => $template_data['language'],
                'response_code' => $responseCode,
                'response_data' => $responseData,
                'raw_data'      => $rawData,
            ], $th);
        }

        $log_data = [
            'response_code'       => $responseCode,
            'category'            => $type,
            'category_id'         => $template_data['campaign_id'] ?? $template_data['template_bot_id'],
            'rel_type'            => $rel_type,
            'rel_id'              => $template_data['rel_id'],
            'category_params'     => json_encode(['templateId' => $template_data['template_id'], 'message' => $message ?? '']),
            'response_data'       => $responseData,
            'raw_data'            => $rawData,
            'phone_number_id'     => get_setting('whatsapp.wm_default_phone_number_id'),
            'access_token'        => get_setting('whatsapp.wm_access_token'),
            'business_account_id' => get_setting('whatsapp.wm_business_account_id'),
        ];

        WmActivityLog::create($log_data);

        return ['status' => $status, 'log_data' => $log_data, 'data' => $data ?? [], 'message' => $message->error->message ?? ''];
    }

    /**
     * Send a message using the WhatsApp Cloud API
     *
     * @param  string      $to           Recipient phone number
     * @param  array       $message_data Data for the message
     * @param  string|null $fromNumber   Optional sender phone number
     * @return array       Response containing status, log data, and any response data or error message
     */
    public function sendMessage($to, $message_data, $fromNumber = null, $folder = 'bot_files')
    {
        $message_data       = parseMessageText($message_data);
        $whatsapp_cloud_api = $this->loadConfig($fromNumber);

        try {
            $rows = [];
            if (! empty($message_data['button1_id'])) {
                $rows[] = new Button($message_data['button1_id'], $message_data['button1']);
            }
            if (! empty($message_data['button2_id'])) {
                $rows[] = new Button($message_data['button2_id'], $message_data['button2']);
            }
            if (! empty($message_data['button3_id'])) {
                $rows[] = new Button($message_data['button3_id'], $message_data['button3']);
            }
            if (! empty($rows)) {
                $action = new ButtonAction($rows);
                $result = $whatsapp_cloud_api->sendButton(
                    $to,
                    $message_data['reply_text'],
                    $action,
                    $message_data['bot_header'],
                    $message_data['bot_footer']
                );
            } elseif (! empty($message_data['button_name']) && ! empty($message_data['button_url']) && filter_var($message_data['button_url'], \FILTER_VALIDATE_URL)) {
                $header = new TitleHeader($message_data['bot_header']);

                $result = $whatsapp_cloud_api->sendCtaUrl(
                    $to,
                    $message_data['button_name'],
                    $message_data['button_url'],
                    $header,
                    $message_data['reply_text'],
                    $message_data['bot_footer'],
                );
            } else {
                $message = $message_data['bot_header'] . "\n" . $message_data['reply_text'] . "\n" . $message_data['bot_footer'];
                if (! empty($message_data['filename'])) {
                    $url            = asset('storage/' . $message_data['filename']);
                    $link_id        = new LinkID($url);
                    $fileExtensions = get_meta_allowed_extension();
                    $extension      = strtolower(pathinfo($message_data['filename'], PATHINFO_EXTENSION));
                    $fileType       = array_key_first(array_filter($fileExtensions, fn ($data) => in_array('.' . $extension, explode(', ', $data['extension']))));
                    if ($fileType == 'image') {
                        $result = $whatsapp_cloud_api->sendImage($to, $link_id, $message);
                    } elseif ($fileType == 'video') {
                        $result = $whatsapp_cloud_api->sendVideo($to, $link_id, $message);
                    } elseif ($fileType == 'document') {
                        $result = $whatsapp_cloud_api->sendDocument($to, $link_id, $message_data['filename'], $message);
                    }
                } else {
                    $result = $whatsapp_cloud_api->sendTextMessage($to, $message, true);
                }
            }

            $status       = true;
            $data         = json_decode($result->body());
            $responseCode = $result->httpStatusCode();
            $responseData = $data;
            $rawData      = json_encode($result->request()->body());
        } catch (\Netflie\WhatsAppCloudApi\Response\ResponseException $th) {
            $status       = false;
            $message      = $th->responseData()['error']['message'] ?? $th->rawResponse() ?? $th->getMessage();
            $responseCode = $th->httpStatusCode();
            $responseData = $message;
            $rawData      = json_encode([]);
        }

        $log_data['response_code']   = $responseCode;
        $log_data['category']        = $folder == 'bot_files' ? 'message_bot' : '';
        $log_data['category_id']     = $message_data['id'] ?? 0;
        $log_data['rel_type']        = $message_data['rel_type'];
        $log_data['rel_id']          = ' - ';
        $log_data['category_params'] = json_encode(['message' => $message ?? '']);
        $log_data['response_data']   = ! empty($responseData) ? json_encode($responseData) : '';
        $log_data['raw_data']        = $rawData;

        $log_data['phone_number_id']     = get_setting('whatsmark.wm_default_phone_number_id');
        $log_data['access_token']        = get_setting('whatsmark.wm_access_token');
        $log_data['business_account_id'] = get_setting('whatsmark.wm_business_account_id');

        WmActivityLog::create($log_data);

        return ['status' => $status, 'log_data' => $log_data ?? [], 'data' => $data ?? [], 'message' => $message->error->message ?? ''];
    }

    /**
     * Send bulk campaign to WhatsApp recipients
     *
     * @param  string      $to           Recipient phone number
     * @param  array       $templateData Template configuration
     * @param  array       $campaign     Campaign data
     * @param  string|null $fromNumber   Sender phone number (optional)
     * @return array       Response data
     */
    public function sendBulkCampaign($to, $templateData, $campaign, $fromNumber = null)
    {
        try {
            // Parse template data for header, body, and buttons
            $headerData = [];
            if ($templateData['header_data_format'] == 'TEXT') {
                $headerData = parseCsvText('header', $templateData, $campaign);
            }

            $bodyData    = parseCsvText('body', $templateData, $campaign);
            $buttonsData = parseCsvText('footer', $templateData, $campaign);

            // Get file link if available
            $fileLink = ($templateData['filename']) ? asset('storage/' . $templateData['filelink']) : '';

            // Build components for WhatsApp message
            $componentHeader  = $this->buildHeaderComponent($templateData, $fileLink, $headerData);
            $componentBody    = $this->buildTextComponent($bodyData);
            $componentButtons = $this->buildTextComponent($buttonsData);

            // Load WhatsApp API configuration
            $whatsappCloudApi = $this->loadConfig($fromNumber);

            // Create components object and send template
            $components = new Component($componentHeader, $componentBody, $componentButtons);
            $result     = $whatsappCloudApi->sendTemplate(
                $to,
                $templateData['template_name'],
                $templateData['language'],
                $components
            );

            return [
                'status'       => true,
                'data'         => json_decode($result->body(), true),
                'responseCode' => $result->httpStatusCode(),
                'message'      => '',
                'phone'        => $to,
            ];
        } catch (ResponseException $e) {

            whatsapp_log('WhatsApp API Error: ' . $e->getMessage(), 'error', [
                'phone'         => $to,
                'template'      => $templateData['template_name'],
                'response_code' => $e->httpStatusCode(),
                'response_data' => $e->responseData() ?? [],
            ], $e);

            return [
                'status'       => false,
                'data'         => [],
                'responseCode' => $e->httpStatusCode(),
                'message'      => $e->responseData()['error']['message'] ?? $e->getMessage(),
                'phone'        => $to,
            ];
        } catch (\Exception $e) {

            whatsapp_log('WhatsApp Campaign Error: ' . $e->getMessage(), 'error', [
                'phone'         => $to,
                'template'      => $templateData['template_name'] ?? 'unknown',
                'response_code' => 500,
            ], $e);

            return [
                'status'       => false,
                'data'         => [],
                'responseCode' => 500,
                'message'      => $e->getMessage(),
                'phone'        => $to,
            ];
        }
    }

    /**
     * Retry sending a campaign message with exponential backoff
     *
     * @param  string      $to           Recipient phone number
     * @param  array       $templateData Template configuration
     * @param  array       $campaign     Campaign data
     * @param  string|null $fromNumber   Sender phone number (optional)
     * @param  int         $maxRetries   Maximum number of retry attempts
     * @return array       Response data
     */
    public function sendWithRetry($to, $templateData, $campaign, $fromNumber = null, $maxRetries = 3)
    {
        $attempt = 0;
        $result  = null;

        while ($attempt < $maxRetries) {
            $result = $this->sendBulkCampaign($to, $templateData, $campaign, $fromNumber);

            // If successful or not a retryable error, break the loop
            if ($result['status'] || ! $this->isRetryableError($result['responseCode'])) {
                break;
            }

            // Exponential backoff: wait longer between each retry
            $waitTime = pow(2, $attempt) * 1000000; // in microseconds (1s, 2s, 4s)
            usleep($waitTime);
            $attempt++;

            whatsapp_log("Retrying WhatsApp message to {$to} (attempt {$attempt})", 'info', [
                'phone'         => $to,
                'template'      => $templateData['template_name'] ?? 'unknown',
                'attempt'       => $attempt,
                'response_code' => $result['responseCode'] ?? null,
                'response_data' => $result['data']         ?? [],
            ]);
        }

        return $result;
    }

    /**
     * Check if an error is retryable
     *
     * @param  int  $statusCode HTTP status code
     * @return bool Whether the error is retryable
     */
    protected function isRetryableError($statusCode)
    {
        // Retry on rate limiting, server errors, and certain client errors
        return in_array($statusCode, [408, 429, 500, 502, 503, 504]);
    }

    /**
     * Handle batch processing for large campaigns
     *
     * @param  array $recipients   List of recipients
     * @param  array $templateData Template configuration
     * @param  int   $batchSize    Batch size (default: 50)
     * @return array Results for each recipient
     */
    public function processBatchCampaign($recipients, $templateData, $batchSize = 50)
    {
        $results = [];
        $batches = array_chunk($recipients, $batchSize);

        foreach ($batches as $batch) {
            foreach ($batch as $recipient) {
                $to        = $recipient['phone'];
                $result    = $this->sendBulkCampaign($to, $templateData, $recipient);
                $results[] = $result;
            }

            // Add a small delay between batches to avoid rate limiting
            if (count($batches) > 1) {
                usleep(500000);
            }
        }

        return $results;
    }

    private function buildHeaderComponent($templateData, $fileLink, $headerData)
    {
        return match ($templateData['header_data_format']) {
            'IMAGE'    => [['type' => 'image', 'image' => ['link' => $fileLink]]],
            'DOCUMENT' => [['type' => 'document', 'document' => ['link' => $fileLink, 'filename' => $templateData['filename']]]],
            'VIDEO'    => [['type' => 'video', 'video' => ['link' => $fileLink]]],
            default    => collect($headerData)->map(fn ($header) => ['type' => 'text', 'text' => $header])->toArray(),
        };
    }

    private function buildTextComponent($data)
    {
        return collect($data)->map(fn ($text) => ['type' => 'text', 'text' => $text])->toArray();
    }

    /**
     * Retrieve a URL for a media file using its media ID
     *
     * @param  string      $media_id    Media ID to retrieve the URL for
     * @param  string      $accessToken Access token for authentication
     * @return string|null Filename of the saved media file or null on failure
     */
    public function retrieveUrl($media_id)
    {
        $url         = self::$facebookAPI . $media_id;
        $accessToken = $this->getToken();

        $response = Http::withToken($accessToken)->get($url);

        if ($response->successful()) {
            $responseData = $response->json();

            if (isset($responseData['url'])) {
                $mediaUrl  = $responseData['url'];
                $mediaData = Http::withToken($accessToken)->get($mediaUrl);

                if ($mediaData->successful()) {
                    $imageContent = $mediaData->body();
                    $contentType  = $mediaData->header('Content-Type');

                    $extensionMap = self::$extensionMap;
                    $extension    = $extensionMap[$contentType] ?? 'unknown';
                    $filename     = 'media_' . uniqid() . '.' . $extension;
                    $storagePath  = 'whatsapp-attachments/' . $filename;

                    Storage::disk('public')->put($storagePath, $imageContent);

                    return $filename;
                }
            }
        }

        return null;
    }

    /**
     * Send a flow message based on node type
     *
     * @param  string $to            Recipient phone number
     * @param  array  $nodeData      Node data from flow
     * @param  string $nodeType      Type of node
     * @param  string $phoneNumberId WhatsApp phone number ID
     * @param  array  $contactData   Contact information
     * @param  array  $context       Flow context information
     * @return array  Response data
     */
    public function sendFlowMessage($to, $nodeData, $nodeType, $phoneNumberId, $contactData, $context = [])
    {
        switch ($nodeType) {
            case 'textMessage':
                return $this->sendFlowTextMessage($to, $nodeData, $phoneNumberId, $contactData, $context);

            case 'buttonMessage':
                return $this->sendFlowButtonMessage($to, $nodeData, $phoneNumberId, $contactData, $context);

            case 'callToAction':
                return $this->sendFlowCTAMessage($to, $nodeData, $phoneNumberId, $contactData, $context);

            case 'templateMessage':
                return $this->sendFlowTemplateMessage($to, $nodeData, $phoneNumberId, $contactData, $context);

            case 'mediaMessage':
                return $this->sendFlowMediaMessage($to, $nodeData, $phoneNumberId, $contactData, $context);

            case 'listMessage':
                return $this->sendFlowListMessage($to, $nodeData, $phoneNumberId, $contactData, $context);

            case 'locationMessage':
                return $this->sendFlowLocationMessage($to, $nodeData, $phoneNumberId, $contactData, $context);

            case 'contactMessage':
                return $this->sendFlowContactMessage($to, $nodeData, $phoneNumberId, $contactData, $context);

            case 'aiAssistant':
                return $this->sendFlowAiMessage($to, $nodeData, $phoneNumberId, $contactData, $context);

            default:
                return ['status' => false, 'message' => 'Unsupported node type: ' . $nodeType];
        }
    }

    protected function sendFlowCTAMessage($to, $nodeData, $phoneNumberId, $contactData, $context)
    {
        // Extract CTA fields from the first output element using array functions
        $output     = collect($nodeData['output'] ?? [])->first() ?? [];
        $header     = $this->replaceFlowVariables($output['bot_header'] ?? '', $contactData);
        $valueText  = $this->replaceFlowVariables($output['reply_text'] ?? '', $contactData);
        $footer     = $this->replaceFlowVariables($output['bot_footer'] ?? '', $contactData);
        $buttonText = $output['buttonText'] ?? 'Click Here';
        $buttonLink = $output['buttonLink'] ?? '';

        try {
            // Use CtaUrl message type from the library
            $whatsapp_cloud_api = $this->loadConfig($phoneNumberId);

            // Create header component if provided
            $headerComponent = null;
            if (! empty($header)) {
                $headerComponent = new TitleHeader($header);
            }

            // Send the CTA message
            $result = $whatsapp_cloud_api->sendCtaUrl(
                $to,
                $buttonText,
                $buttonLink,
                $headerComponent,
                $valueText,
                $footer
            );

            $response = [
                'status'       => true,
                'data'         => json_decode($result->body()),
                'responseCode' => $result->httpStatusCode(),
                'responseData' => $result->decodedBody(),
            ];

            // Log the activity
            $this->logFlowActivity($to, $response, [
                'rel_type'    => $contactData->type ?? 'guest',
                'rel_id'      => $contactData->id   ?? '',
                'header'      => $header,
                'value_text'  => $valueText,
                'footer'      => $footer,
                'button_text' => $buttonText,
                'button_link' => $buttonLink,
            ], 'flow_cta_message');

            return $response;
        } catch (\Throwable $e) {
            whatsapp_log('Flow CTA message error', 'error', [
                'error' => $e->getMessage(),
                'to'    => $to,
            ], $e);

            return [
                'status'  => false,
                'message' => $e->getMessage(),
            ];
        }
    }

    /**
     * Send a text message from flow
     */
    protected function sendFlowTextMessage($to, $nodeData, $phoneNumberId, $contactData, $context)
    {
        // Use array functions to get reply_text from output if present
        $replyText = '';
        if (! empty($nodeData['output']) && is_array($nodeData['output'])) {
            $replyText = collect($nodeData['output'])
                ->pluck('reply_text')
                ->filter()
                ->first() ?? '';
        }
        if (empty($replyText)) {
            // Fallback to 'message' key if present
            $replyText = $nodeData['message'] ?? '';
        }
        // Replace variables in message
        $message = $this->replaceFlowVariables($replyText, $contactData);

        $messageData = [
            'rel_type'   => $contactData->type ?? 'guest',
            'rel_id'     => $contactData->id   ?? '',
            'reply_text' => $message,
            'bot_header' => '',
            'bot_footer' => '',
        ];

        return $this->sendMessage($to, $messageData, $phoneNumberId);
    }

    /**
     * Send a button message from flow
     */
    protected function sendFlowButtonMessage($to, $nodeData, $phoneNumberId, $contactData, $context)
    {
        // Extract button message fields from the first output element
        $output    = collect($nodeData['output'] ?? [])->first() ?? [];
        $replyText = $this->replaceFlowVariables($output['reply_text'] ?? '', $contactData);
        $button1   = $output['button1'] ?? null;
        $button2   = $output['button2'] ?? null;
        $button3   = $output['button3'] ?? null;

        // Prepare buttons data with UNIQUE IDs
        $buttons       = [];
        $currentNodeId = $context['current_node'] ?? uniqid();

        if ($button1) {
            $uniqueButtonId = $currentNodeId . '_btn_0';
            $buttons[]      = new Button($uniqueButtonId, $button1);
        }
        if ($button2) {
            $uniqueButtonId = $currentNodeId . '_btn_1';
            $buttons[]      = new Button($uniqueButtonId, $button2);
        }
        if ($button3) {
            $uniqueButtonId = $currentNodeId . '_btn_2';
            $buttons[]      = new Button($uniqueButtonId, $button3);
        }

        if (empty($buttons)) {
            return [
                'status'  => false,
                'message' => 'No buttons defined for button message',
            ];
        }

        try {
            $whatsapp_cloud_api = $this->loadConfig($phoneNumberId);

            // Create button action
            $buttonAction = new ButtonAction($buttons);

            // Send the button message
            $result = $whatsapp_cloud_api->sendButton(
                $to,
                $replyText,
                $buttonAction
            );

            $response = [
                'status'       => true,
                'data'         => json_decode($result->body()),
                'responseCode' => $result->httpStatusCode(),
                'responseData' => $result->decodedBody(),
            ];

            // Log the activity with unique button IDs
            $buttonTexts = array_filter([$button1, $button2, $button3]);
            $this->logFlowActivity($to, $response, [
                'rel_type'   => $contactData->type ?? 'guest',
                'rel_id'     => $contactData->id   ?? '',
                'reply_text' => $replyText,
                'buttons'    => $buttonTexts,
                'next_nodes' => $context['next_nodes'] ?? [],
            ], 'flow_button_message');

            return $response;
        } catch (\Throwable $e) {
            whatsapp_log('Flow button message error', 'error', [
                'error' => $e->getMessage(),
                'to'    => $to,
            ], $e);

            return [
                'status'  => false,
                'message' => $e->getMessage(),
            ];
        }
    }

    /**
     * Send a template message from flow
     */
    protected function sendFlowTemplateMessage($to, $nodeData, $phoneNumberId, $contactData, $context)
    {
        $templateId = $nodeData['templateId'] ?? '';
        $params     = $nodeData['params']     ?? [];

        // Get template details from database
        $template = \App\Models\WhatsappTemplate::where('template_id', $templateId)->first();

        if (! $template) {
            return ['status' => false, 'message' => 'Template not found'];
        }

        // Prepare template data
        $templateData = [
            'rel_type'           => $contactData->type ?? 'guest',
            'rel_id'             => $contactData->id   ?? '',
            'template_id'        => $templateId,
            'template_name'      => $template->template_name,
            'language'           => $template->language,
            'header_data_format' => $template->header_data_format,
            'header_data_text'   => $template->header_data_text,
            'body_data'          => $template->body_data,
            'footer_data'        => $template->footer_data,
            'template_bot_id'    => 0,
        ];

        // Add parameters
        if (! empty($params['header'])) {
            $templateData['header_params'] = json_encode($this->replaceFlowVariablesInArray($params['header'], $contactData));
        }

        if (! empty($params['body'])) {
            $templateData['body_params'] = json_encode($this->replaceFlowVariablesInArray($params['body'], $contactData));
        }

        if (! empty($params['footer'])) {
            $templateData['footer_params'] = json_encode($this->replaceFlowVariablesInArray($params['footer'], $contactData));
        }

        return $this->sendTemplate($to, $templateData, 'template_bot', $phoneNumberId);
    }

    /**
     * Send a media message from flow
     */
    protected function sendFlowMediaMessage($to, $nodeData, $phoneNumberId, $contactData, $context)
    {
        // Extract media fields from the first output element using array functions
        $output    = collect($nodeData['output'] ?? [])->first() ?? [];
        $mediaType = $output['media_type']                       ?? 'image';
        $mediaUrl  = $output['media_url']                        ?? '';
        $caption   = $this->replaceFlowVariables($output['media_caption'] ?? '', $contactData);
        $fileName  = $output['media_filename'] ?? basename($mediaUrl);

        // Load WhatsApp Cloud API
        $whatsapp_cloud_api = $this->loadConfig($phoneNumberId);

        try {
            $result = null;
            $link   = new LinkID($mediaUrl);

            switch ($mediaType) {
                case 'image':
                    $result = $whatsapp_cloud_api->sendImage($to, $link, $caption);
                    break;

                case 'video':
                    $result = $whatsapp_cloud_api->sendVideo($to, $link, $caption);
                    break;

                case 'audio':
                    $result = $whatsapp_cloud_api->sendAudio($to, $link);
                    break;

                case 'document':
                    $result = $whatsapp_cloud_api->sendDocument($to, $link, $fileName, $caption);
                    break;

                default:
                    return ['status' => false, 'message' => 'Unsupported media type'];
            }

            $response = [
                'status'       => true,
                'data'         => json_decode($result->body()),
                'responseCode' => $result->httpStatusCode(),
                'responseData' => $result->decodedBody(),
            ];

            // Log the activity
            $this->logFlowActivity($to, $response, [
                'rel_type'   => $contactData->type ?? 'guest',
                'rel_id'     => $contactData->id   ?? '',
                'media_type' => $mediaType,
                'media_url'  => $mediaUrl,
                'caption'    => $caption,
            ], 'flow_media_message');

            return $response;
        } catch (\Throwable $e) {
            whatsapp_log('Flow media message error', 'error', [
                'error'      => $e->getMessage(),
                'media_type' => $mediaType,
                'to'         => $to,
            ], $e);

            return [
                'status'  => false,
                'message' => $e->getMessage(),
            ];
        }
    }

    /**
     * Send a list message from flow
     */
    protected function sendFlowListMessage($to, $nodeData, $phoneNumberId, $contactData, $context)
    {
        // Extract list message fields from the first output element
        $output     = collect($nodeData['output'] ?? [])->first() ?? [];
        $headerText = $this->replaceFlowVariables($output['bot_header'] ?? '', $contactData);
        $bodyText   = $this->replaceFlowVariables($output['reply_text'] ?? '', $contactData);
        $footerText = $this->replaceFlowVariables($output['bot_footer'] ?? '', $contactData);
        $buttonText = $output['buttonText'] ?? 'View Options';
        $sections   = $output['sections']   ?? [];

        if (empty($sections)) {
            return [
                'status'  => false,
                'message' => 'No sections defined for list message',
            ];
        }

        // Load WhatsApp Cloud API
        $whatsapp_cloud_api = $this->loadConfig($phoneNumberId);
        $currentNodeId      = $context['current_node'] ?? uniqid();

        try {
            // Format sections for WhatsApp API with unique IDs
            $formattedSections = [];
            foreach ($sections as $sectionIndex => $section) {
                $formattedSection = [
                    'title' => $section['title'] ?? 'Section ' . ($sectionIndex + 1),
                    'rows'  => [],
                ];

                foreach ($section['items'] as $itemIndex => $item) {
                    // Create unique list item ID for flow navigation
                    $uniqueItemId = $currentNodeId . '_item_' . $sectionIndex . '_' . $itemIndex;

                    $formattedSection['rows'][] = [
                        'id'          => $uniqueItemId, // Use unique ID instead of original item ID
                        'title'       => $item['title']       ?? 'Item ' . ($itemIndex + 1),
                        'description' => $item['description'] ?? '',
                    ];
                }
                $formattedSections[] = $formattedSection;
            }

            // Create interactive list message payload
            $interactivePayload = [
                'messaging_product' => 'whatsapp',
                'recipient_type'    => 'individual',
                'to'                => $to,
                'type'              => 'interactive',
                'interactive'       => [
                    'type' => 'list',
                    'body' => [
                        'text' => $bodyText,
                    ],
                    'action' => [
                        'button'   => $buttonText,
                        'sections' => $formattedSections,
                    ],
                ],
            ];

            // Add header if provided
            if (! empty($headerText)) {
                $interactivePayload['interactive']['header'] = [
                    'type' => 'text',
                    'text' => $headerText,
                ];
            }

            // Add footer if provided
            if (! empty($footerText)) {
                $interactivePayload['interactive']['footer'] = [
                    'text' => $footerText,
                ];
            }

            whatsapp_log('Sending list message with unique IDs', 'debug', [
                'node_id'        => $currentNodeId,
                'to'             => $to,
                'sections_count' => count($formattedSections),
                'total_items'    => array_sum(array_map(function ($section) {
                    return count($section['rows']);
                }, $formattedSections)),
            ]);

            // Send using raw API call
            $response = Http::withToken($this->getToken())
                ->post(self::getBaseUrl() . $this->getPhoneID() . '/messages', $interactivePayload);

            $responseData = $response->json();

            $result = [
                'status'       => $response->successful(),
                'data'         => (object) $responseData,
                'responseCode' => $response->status(),
                'responseData' => $responseData,
            ];

            whatsapp_log('List message sent', 'info', [
                'node_id'       => $currentNodeId,
                'to'            => $to,
                'response_code' => $response->status(),
                'success'       => $response->successful(),
                'message_id'    => $responseData['messages'][0]['id'] ?? 'unknown',
            ]);

            // Log the activity
            $this->logFlowActivity($to, $result, [
                'rel_type'        => $contactData->type ?? 'guest',
                'rel_id'          => $contactData->id   ?? '',
                'header'          => $headerText,
                'body'            => $bodyText,
                'footer'          => $footerText,
                'button'          => $buttonText,
                'sections'        => $sections,
                'unique_ids_used' => true,
                'next_nodes'      => $context['next_nodes'] ?? [],
            ], 'flow_list_message');

            return $result;
        } catch (\Throwable $e) {
            whatsapp_log('Flow list message error', 'error', [
                'error'   => $e->getMessage(),
                'to'      => $to,
                'node_id' => $currentNodeId,
                'trace'   => $e->getTraceAsString(),
            ], $e);

            return [
                'status'  => false,
                'message' => $e->getMessage(),
            ];
        }
    }

    /**
     * Send a location message from flow
     */
    protected function sendFlowLocationMessage($to, $nodeData, $phoneNumberId, $contactData, $context)
    {
        // Extract location fields from the first output element using array functions
        $output    = collect($nodeData['output'] ?? [])->first() ?? [];
        $latitude  = $output['location_latitude']                ?? '';
        $longitude = $output['location_longitude']               ?? '';
        $name      = $this->replaceFlowVariables($output['location_name'] ?? '', $contactData);
        $address   = $this->replaceFlowVariables($output['location_address'] ?? '', $contactData);

        // In netflie/whatsapp-cloud-api v2.2, we may need to use raw API for location
        try {
            // Send using raw API call since the library might not support location directly
            $response = Http::withToken($this->getToken())
                ->post(self::getBaseUrl() . $this->getPhoneID() . '/messages', [
                    'messaging_product' => 'whatsapp',
                    'recipient_type'    => 'individual',
                    'to'                => $to,
                    'type'              => 'location',
                    'location'          => [
                        'latitude'  => (float) $latitude,
                        'longitude' => (float) $longitude,
                        'name'      => $name,
                        'address'   => $address,
                    ],
                ]);

            $responseData = $response->json();

            $result = [
                'status'       => $response->successful(),
                'data'         => (object) $responseData,
                'responseCode' => $response->status(),
                'responseData' => $responseData,
            ];

            // Log the activity
            $this->logFlowActivity($to, $result, [
                'rel_type'  => $contactData->type ?? 'guest',
                'rel_id'    => $contactData->id   ?? '',
                'latitude'  => $latitude,
                'longitude' => $longitude,
                'name'      => $name,
                'address'   => $address,
            ], 'flow_location_message');

            return $result;
        } catch (\Throwable $e) {
            whatsapp_log('Flow location message error', 'error', [
                'error' => $e->getMessage(),
                'to'    => $to,
            ], $e);

            return [
                'status'  => false,
                'message' => $e->getMessage(),
            ];
        }
    }

    private function extractButtonIdFromMessage($message)
    {
        $buttonId = null;

        // Extract button ID if this is a button/list response
        if (! empty($message['interactive'])) {
            if ($message['interactive']['type'] == 'button_reply') {
                $buttonId = $message['interactive']['button_reply']['id'];

                whatsapp_log('Button response extracted', 'debug', [
                    'button_id'    => $buttonId,
                    'button_title' => $message['interactive']['button_reply']['title'] ?? 'unknown',
                ]);
            } elseif ($message['interactive']['type'] == 'list_reply') {
                // For list replies, use the ID (which now contains our unique format)
                $buttonId = $message['interactive']['list_reply']['id'];

                whatsapp_log('List response extracted', 'debug', [
                    'list_item_id'     => $buttonId,
                    'list_title'       => $message['interactive']['list_reply']['title']       ?? 'unknown',
                    'list_description' => $message['interactive']['list_reply']['description'] ?? 'unknown',
                ]);
            }
        }

        return $buttonId;
    }

    /**
     * Send a contact message from flow
     */
    protected function sendFlowContactMessage($to, $nodeData, $phoneNumberId, $contactData, $context)
    {
        // Extract contacts array from the first output element using array functions
        $output            = collect($nodeData['output'] ?? [])->first() ?? [];
        $contacts          = $output['contacts']                         ?? [];
        $processedContacts = [];

        // Process and replace variables in contact data
        foreach ($contacts as $contact) {
            $firstName     = $this->replaceFlowVariables($contact['firstName'] ?? '', $contactData);
            $lastName      = $this->replaceFlowVariables($contact['lastName'] ?? '', $contactData);
            $formattedName = trim($firstName . ' ' . $lastName);
            $processed     = [
                'name' => [
                    'formatted_name' => $formattedName,
                    'first_name'     => $firstName,
                    'last_name'      => $lastName,
                ],
                'phones' => [
                    [
                        'phone' => $this->replaceFlowVariables($contact['phone'] ?? '', $contactData),
                        'type'  => 'CELL',
                    ],
                ],
            ];

            // Add email if provided
            if (! empty($contact['email'])) {
                $processed['emails'] = [
                    [
                        'email' => $this->replaceFlowVariables($contact['email'] ?? '', $contactData),
                        'type'  => 'WORK',
                    ],
                ];
            }

            // Add company info if provided
            if (! empty($contact['company'])) {
                $processed['org'] = [
                    'company' => $this->replaceFlowVariables($contact['company'] ?? '', $contactData),
                ];

                if (! empty($contact['title'])) {
                    $processed['org']['title'] = $this->replaceFlowVariables($contact['title'] ?? '', $contactData);
                }
            }

            $processedContacts[] = $processed;
        }

        // In netflie/whatsapp-cloud-api v2.2, we need to use raw API for contacts
        try {
            // Send using raw API call
            $response = Http::withToken($this->getToken())
                ->post(self::getBaseUrl() . $this->getPhoneID() . '/messages', [
                    'messaging_product' => 'whatsapp',
                    'recipient_type'    => 'individual',
                    'to'                => $to,
                    'type'              => 'contacts',
                    'contacts'          => $processedContacts,
                ]);

            $responseData = $response->json();

            $result = [
                'status'       => $response->successful(),
                'data'         => (object) $responseData,
                'responseCode' => $response->status(),
                'responseData' => $responseData,
            ];

            // Log the activity
            $this->logFlowActivity($to, $result, [
                'rel_type' => $contactData->type ?? 'guest',
                'rel_id'   => $contactData->id   ?? '',
                'contacts' => $processedContacts,
            ], 'flow_contact_message');

            return $result;
        } catch (\Throwable $e) {
            whatsapp_log('Flow contact message error', 'error', [
                'error' => $e->getMessage(),
                'to'    => $to,
            ], $e);

            return [
                'status'  => false,
                'message' => $e->getMessage(),
            ];
        }
    }

    /**
     * Send an AI-generated message from flow
     */
    protected function sendFlowAiMessage($to, $nodeData, $phoneNumberId, $contactData, $context)
    {
        $prompt      = $this->replaceFlowVariables($nodeData['prompt'] ?? '', $contactData);
        $aiModel     = $nodeData['aiModel']     ?? 'gpt-3.5-turbo';
        $contextType = $nodeData['contextType'] ?? 'message';

        // Get AI response (implement this based on your AI service)
        $aiResponse = $this->generateFlowAiResponse($prompt, $aiModel, $contextType, $context);

        if (! $aiResponse) {
            return ['status' => false, 'message' => 'Failed to generate AI response'];
        }

        $messageData = [
            'rel_type'   => $contactData->type ?? 'guest',
            'rel_id'     => $contactData->id   ?? '',
            'reply_text' => $aiResponse,
            'bot_header' => '',
            'bot_footer' => '',
        ];

        return $this->sendMessage($to, $messageData, $phoneNumberId);
    }

    /**
     * Generate AI response for flow
     */
    protected function generateFlowAiResponse($prompt, $model, $contextType, $flowContext)
    {
        // Check if AI is enabled in settings
        if (! get_setting('whats-mark.enable_openai_in_chat')) {
            return 'AI response generation is disabled. Please enable it in settings.';
        }

        $apiKey = get_setting('whats-mark.openai_secret_key');
        if (empty($apiKey)) {
            return 'AI API key not configured.';
        }

        try {
            // Get conversation context based on contextType
            $contextPrompt = '';
            if ($contextType === 'conversation' && ! empty($flowContext['chat_id'])) {
                // Get recent messages for context
                $recentMessages = \App\Models\ChatMessage::where('interaction_id', $flowContext['chat_id'])
                    ->orderBy('created_at', 'desc')
                    ->limit(5)
                    ->get()
                    ->reverse();

                foreach ($recentMessages as $msg) {
                    $contextPrompt .= ($msg->staff_id ? 'Agent: ' : 'User: ') . strip_tags($msg->message) . "\n";
                }
            }

            // Make request to OpenAI API
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $apiKey,
                'Content-Type'  => 'application/json',
            ])->post('https://api.openai.com/v1/chat/completions', [
                'model'    => $model,
                'messages' => [
                    ['role' => 'system', 'content' => 'You are a helpful assistant that generates concise and helpful responses.'],
                    ['role' => 'user', 'content' => $contextPrompt . "\n" . $prompt],
                ],
                'max_tokens'  => 500,
                'temperature' => 0.7,
            ]);

            $data = $response->json();

            if (isset($data['choices'][0]['message']['content'])) {
                return $data['choices'][0]['message']['content'];
            }

            return 'Failed to generate response: ' . ($data['error']['message'] ?? 'Unknown error');
        } catch (\Throwable $e) {
            whatsapp_log('AI response generation error', 'error', [
                'error' => $e->getMessage(),
                'stack' => $e->getTraceAsString(),
            ], $e);

            return 'Error generating AI response: ' . $e->getMessage();
        }
    }

    /**
     * Replace flow variables in text with contact data
     */
    public function replaceFlowVariables($text, $contactData)
    {
        if (empty($text)) {
            return $text;
        }
        $data['rel_type']   = $contactData['type'] ?? 'lead';
        $data['rel_id']     = $contactData['id']   ?? '';
        $data['reply_text'] = $text;

        $data = parseMessageText($data);

        return $data['reply_text'] ?? '';
    }

    /**
     * Replace variables in an array of values
     */
    public function replaceFlowVariablesInArray($array, $contactData)
    {
        return array_map(function ($item) use ($contactData) {
            return $this->replaceFlowVariables($item, $contactData);
        }, $array);
    }

    /**
     * Log flow activity to database
     */
    protected function logFlowActivity($to, $response, $messageData, $category)
    {
        $data = [
            'response_code'       => $response['responseCode'] ?? 200,
            'category'            => $category,
            'category_id'         => $messageData['flow_id']  ?? 0,
            'rel_type'            => $messageData['rel_type'] ?? '',
            'rel_id'              => $messageData['rel_id']   ?? '',
            'category_params'     => json_encode($messageData),
            'response_data'       => json_encode($response['responseData'] ?? []),
            'raw_data'            => json_encode($response['data'] ?? []),
            'phone_number_id'     => get_setting('whatsapp.wm_default_phone_number_id'),
            'access_token'        => get_setting('whatsapp.wm_access_token'),
            'business_account_id' => get_setting('whatsapp.wm_business_account_id'),
        ];

        WmActivityLog::create($data);
    }

    protected function storeFlowMessage($response, $chatId, $contactNumber, $contactData, $nodeType, $nodeData, $context)
    {
        if (! isset($response['data']) || ! isset($response['data']->messages[0]->id)) {
            return false;
        }

        $messageId = $response['data']->messages[0]->id;

        // Format message content based on node type
        $messageContent = $this->formatMessageContent($nodeType, $nodeData, $contactData);

        // Ensure we properly store all context information
        // Especially the next_nodes mapping for interactive messages
        $contextToStore = $context;

        // Add additional meta information about the node type
        $contextToStore['node_type'] = $nodeType;

        // Add button/list mappings if applicable
        if (($nodeType === 'buttonMessage' && ! empty($nodeData['buttons'])) || ($nodeType === 'listMessage' && ! empty($nodeData['sections']))
        ) {
            // Make sure we preserve the connection information
            $contextToStore['interactive_type'] = $nodeType;
        }

        $chatMessage = [
            'interaction_id' => $chatId,
            'sender_id'      => get_setting('whatsapp.wm_default_phone_number'),
            'url'            => null,
            'message'        => $messageContent,
            'status'         => 'sent',
            'time_sent'      => now()->toDateTimeString(),
            'message_id'     => $messageId,
            'staff_id'       => 0,
            'type'           => 'text',
            'flow_context'   => json_encode($contextToStore),
        ];

        $messageId = ChatMessage::insertGetId($chatMessage);

        // Send pusher notification if enabled
        if (
            ! empty(get_setting('pusher.app_key')) && ! empty(get_setting('pusher.app_secret')) && ! empty(get_setting('pusher.app_id')) && ! empty(get_setting('pusher.cluster'))
        ) {

            $pusherService = new PusherService;
            $pusherService->trigger('whatsmark-chat-channel', 'whatsmark-chat-event', [
                'chat' => \App\Http\Controllers\WhatsApp\ChatController::newChatMessage($chatId, $messageId),
            ]);
        }

        return $messageId;
    }

    // This method ensures we properly parse interactive responses from WhatsApp
    private function getInteractiveResponseHandler($message, $flowContext)
    {
        if (empty($message['interactive'])) {
            return null;
        }

        $interactiveType = $message['interactive']['type'] ?? null;
        $flowNodeType    = $flowContext['node_type']       ?? null;
        $nextNodes       = $flowContext['next_nodes']      ?? [];

        if ($interactiveType === 'button_reply' && $flowNodeType === 'buttonMessage') {
            $buttonReply = $message['interactive']['button_reply'] ?? null;
            if ($buttonReply && isset($buttonReply['id']) && isset($nextNodes[$buttonReply['id']])) {
                return $nextNodes[$buttonReply['id']];
            }
        }

        if ($interactiveType === 'list_reply' && $flowNodeType === 'listMessage') {
            $listReply = $message['interactive']['list_reply'] ?? null;
            if ($listReply && isset($listReply['description']) && isset($nextNodes[$listReply['description']])) {
                return $nextNodes[$listReply['description']];
            }
        }

        return null;
    }

    // Improved extractTriggerMessage to handle interactive responses better
    private function extractTriggerMessage($message)
    {
        // Handle different message types
        if (isset($message['button']['text'])) {
            return $message['button']['text'];
        } elseif (isset($message['text']['body'])) {
            return $message['text']['body'];
        } elseif (! empty($message['interactive'])) {
            if ($message['interactive']['type'] == 'button_reply') {
                return $message['interactive']['button_reply']['id'];
            } elseif ($message['interactive']['type'] == 'list_reply') {
                // For list replies, use description if available as it matches our context keys
                if (isset($message['interactive']['list_reply']['description'])) {
                    return $message['interactive']['list_reply']['description'];
                }

                return $message['interactive']['list_reply']['id'];
            }
        }

        return '';
    }

    /**
     * Format message content for display in chat UI
     */
    private function formatMessageContent($nodeType, $nodeData, $contactData)
    {
        switch ($nodeType) {
            case 'textMessage':
                $message = $this->replaceFlowVariables($nodeData['message'] ?? '', $contactData);

                return '<p>' . nl2br(e($message)) . '</p>';

            case 'buttonMessage':
                $message = $this->replaceFlowVariables($nodeData['message'] ?? '', $contactData);
                $buttons = $nodeData['buttons'] ?? [];

                $buttonHtml = "<div class='flex flex-col mt-2 space-y-2'>";
                foreach ($buttons as $button) {
                    $buttonHtml .= "<button class='bg-gray-100 text-success-500 px-3 py-2 rounded-lg flex items-center justify-center text-xs space-x-2 w-full
                    dark:bg-gray-800 dark:text-success-400'>" . e($button['text']) . '</button>';
                }
                $buttonHtml .= '</div>';

                return '<p>' . nl2br(e($message)) . '</p>' . $buttonHtml;
                break;

            case 'callToAction':
                $output     = collect($nodeData['output'] ?? [])->first() ?? [];
                $header     = $this->replaceFlowVariables($output['bot_header'] ?? '', $contactData);
                $valueText  = $this->replaceFlowVariables($output['reply_text'] ?? '', $contactData);
                $footer     = $this->replaceFlowVariables($output['bot_footer'] ?? '', $contactData);
                $buttonText = $output['buttonText'] ?? 'Click Here';
                $buttonLink = $output['buttonLink'] ?? '#';
                $ctaHtml    = '';
                if (! empty($header)) {
                    $ctaHtml .= "<div class='font-medium mb-2'>" . e($header) . '</div>';
                }
                $ctaHtml .= '<p>' . nl2br(e($valueText)) . '</p>';
                if (! empty($footer)) {
                    $ctaHtml .= "<div class='text-gray-500 text-sm mt-1'>" . e($footer) . '</div>';
                }
                $ctaHtml .= "<a href='" . e($buttonLink) . "' class='bg-info-500 text-white px-3 py-2 rounded-lg flex items-center justify-center text-xs space-x-2 w-full mt-2'>" . e($buttonText) . '</a>';

                return $ctaHtml;
                break;

            case 'mediaMessage':
                $output    = collect($nodeData['output'] ?? [])->first() ?? [];
                $mediaType = $output['media_type']                       ?? 'image';
                $mediaUrl  = $output['media_url']                        ?? '';
                $caption   = $this->replaceFlowVariables($output['media_caption'] ?? '', $contactData);
                $fileName  = $output['media_filename'] ?? basename($mediaUrl);
                $mediaHtml = '';
                if ($mediaType === 'image') {
                    $mediaHtml = "<a href='" . e($mediaUrl) . "' data-lightbox='image-group'><img src='" . e($mediaUrl) . "' class='rounded-lg w-full mb-2'></a>";
                } elseif ($mediaType === 'video') {
                    $mediaHtml = "<video src='" . e($mediaUrl) . "' controls class='rounded-lg w-full'></video>";
                } elseif ($mediaType === 'audio') {
                    $mediaHtml = "<audio controls class='w-64'><source src='" . e($mediaUrl) . "' type='audio/mpeg'></audio>";
                } elseif ($mediaType === 'document') {
                    $mediaHtml = "<a href='" . e($mediaUrl) . "' target='_blank' class='bg-gray-100 text-success-500 px-3 py-2 rounded-lg flex items-center justify-center text-xs space-x-2 w-full dark:bg-gray-800 dark:text-success-400'>" . e($fileName) . '</a>';
                }

                return $mediaHtml . '<p>' . nl2br(e($caption)) . '</p>';
                break;

            case 'listMessage':
                $headerText = $this->replaceFlowVariables($nodeData['headerText'] ?? '', $contactData);
                $bodyText   = $this->replaceFlowVariables($nodeData['bodyText'] ?? '', $contactData);
                $footerText = $this->replaceFlowVariables($nodeData['footerText'] ?? '', $contactData);
                $buttonText = $nodeData['buttonText'] ?? 'View Options';
                $listHtml   = '';
                if (! empty($headerText)) {
                    $listHtml .= "<div class='font-medium mb-2'>" . e($headerText) . '</div>';
                }
                $listHtml .= '<p>' . nl2br(e($bodyText)) . '</p>';
                if (! empty($footerText)) {
                    $listHtml .= "<div class='text-gray-500 text-sm mt-1'>" . e($footerText) . '</div>';
                }
                $listHtml .= "<div class='bg-gray-100 text-success-500 px-3 py-2 rounded-lg flex items-center justify-center text-xs space-x-2 w-full mt-2 dark:bg-gray-800 dark:text-success-400'>" . e($buttonText) . '</div>';

                return $listHtml;
                break;

            case 'locationMessage':
                $name    = $this->replaceFlowVariables($nodeData['name'] ?? '', $contactData);
                $address = $this->replaceFlowVariables($nodeData['address'] ?? '', $contactData);

                return "<div class='bg-gray-100 p-3 rounded-lg'><div class='font-medium'>" . e($name) . "</div><div class='text-sm text-gray-500'>" . e($address) . "</div><div class='text-xs text-info-500 mt-1'>Location: " . e($nodeData['latitude']) . ', ' . e($nodeData['longitude']) . '</div></div>';
                break;

            case 'contactMessage':
                $contacts    = $nodeData['contacts'] ?? [];
                $contactHtml = "<div class='space-y-2'>";
                foreach ($contacts as $contact) {
                    $firstName = $this->replaceFlowVariables($contact['firstName'] ?? '', $contactData);
                    $lastName  = $this->replaceFlowVariables($contact['lastName'] ?? '', $contactData);
                    $phone     = $this->replaceFlowVariables($contact['phone'] ?? '', $contactData);
                    $contactHtml .= "<div class='bg-gray-100 p-3 rounded-lg'><div class='font-medium'>" . e($firstName) . ' ' . e($lastName) . "</div><div class='text-sm'>" . e($phone) . '</div>';
                    if (! empty($contact['email'])) {
                        $email = $this->replaceFlowVariables($contact['email'], $contactData);
                        $contactHtml .= "<div class='text-sm'>" . e($email) . '</div>';
                    }
                    if (! empty($contact['company'])) {
                        $company = $this->replaceFlowVariables($contact['company'], $contactData);
                        $contactHtml .= "<div class='text-sm text-gray-500'>" . e($company) . '</div>';
                    }
                    $contactHtml .= '</div>';
                }
                $contactHtml .= '</div>';

                return $contactHtml;
                break;

            case 'aiAssistant':
                $prompt = $this->replaceFlowVariables($nodeData['prompt'] ?? '', $contactData);

                return "<p class='text-sm italic text-gray-500'>AI response for: " . e($prompt) . '</p>';
                break;

            default:
                return '<p>Unsupported message type</p>';
                break;
        }
    }
}
