<?php

namespace App\Http\Controllers\WhatsApp;

use App\Http\Controllers\Controller;
use App\Models\Campaign;
use App\Models\CampaignDetail;
use App\Models\Contact;
use App\Models\Source;
use App\Models\Status;
use App\Models\WhatsappTemplate;
use App\Rules\PurifiedInput;
use App\Services\MergeFields;
use App\Traits\WhatsApp;
use Carbon\Carbon;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

class ManageCampaign extends Controller
{
    use WhatsApp;
    // =================================================================
    // MAIN CRUD METHODS (Standard RESTful Routes)
    // =================================================================

    /**
     * Display campaigns listing page
     */
    public function index()
    {
        // Check permissions
        if (! checkPermission(['campaigns.view'])) {
            session()->flash('notification', ['type' => 'danger', 'message' => t('access_denied_note')]);

            return redirect()->route('admin.dashboard');
        }

        $campaigns = Campaign::query()
            ->select('campaigns.*', 'whatsapp_templates.template_name as template_name') // or other template columns
            ->leftJoin('whatsapp_templates', 'campaigns.template_id', '=', 'whatsapp_templates.id')
            ->when(! Auth::user()->is_admin && checkPermission('campaigns.view_own'), function ($query) {
                return $query->where('campaigns.created_by', auth()->id());
            })
            ->orderBy('campaigns.created_at', 'desc')
            ->paginate(20);

        $templates = WhatsappTemplate::all();

        // Fetch statuses
        $statuses = Status::select('id', 'name')->orderBy('name')->get();

        // Fetch sources
        $sources     = Source::select('id', 'name')->orderBy('name')->get();
        $mergeFields = $this->getMergeFieldsData();

        return view('chat.manageCampaign', compact(['campaigns', 'templates', 'statuses', 'sources', 'mergeFields']));
    }

    /**
     * Show campaign creation form
     */
    public function create()
    {
        // Check permissions
        if (! checkPermission(['campaigns.create'])) {
            session()->flash('notification', ['type' => 'danger', 'message' => t('access_denied_note')]);

            return redirect()->route('admin.dashboard');
        }

        // Check WhatsApp connection status
        if ($this->isWhatsAppDisconnected()) {
            return view('admin.campaigns.disconnected');
        }

        // Load initial data for form
        $formData = $this->getFormInitialData();

        return view('admin.campaign.create', $formData);
    }

    /**
     * Store new campaign
     */
    public function store(Request $request): JsonResponse|RedirectResponse
    {

        // Check permissions
        if (! checkPermission(['campaigns.create'])) {
            if ($request->ajax()) {
                return response()->json(['success' => false, 'message' => t('access_denied')], 403);
            }

            return redirect()->route('admin.dashboard')->with('error', t('access_denied_note'));
        }

        try {
            DB::beginTransaction();

            // Validate the request

            $validatedData = $this->validateCampaignData($request);
            // Handle file upload if present
            $filename = $this->handleCampaignFileUpload($request);

            // Create campaign
            $campaign = $this->createCampaign($validatedData, $filename);

            // Create campaign details for contacts
            $this->createCampaignDetails($campaign, $validatedData);

            DB::commit();

            if ($request->ajax()) {
                return response()->json([
                    'success'  => true,
                    'message'  => t('campaign_created_successfully'),
                    'redirect' => route('admin.campaigns.list'),
                ]);
            }

            return redirect()->route('admin.campaigns.list')
                ->with('success', t('campaign_created_successfully'));
        } catch (\Exception $e) {
            DB::rollBack();

            Log::error('Campaign creation failed', [
                'error'        => $e->getMessage(),
                'user_id'      => auth()->id(),
                'request_data' => $request->except(['file']),
            ]);

            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => t('campaign_save_failed') . ': ' . $e->getMessage(),
                ], 500);
            }

            return back()->withInput()
                ->with('error', t('campaign_save_failed') . ': ' . $e->getMessage());
        }
    }

    /**
     * Show campaign edit form
     */
    public function edit(int $id)
    {
        // Check permissions
        if (! checkPermission(['campaigns.edit'])) {
            return redirect()->route('admin.dashboard')->with('error', t('access_denied_note'));
        }

        // Find campaign
        $campaign = Campaign::with(['details.contact'])->findOrFail($id);

        // Check ownership if not admin
        if (! Auth::user()->is_admin && $campaign->created_by !== auth()->id()) {

            return redirect()->route('admin.campaign')
                ->with('error', t('access_denied_note'));
        }

        // Load form data with campaign
        $formData = $this->getFormInitialData($campaign);

        return view('chat.manageCampaign', array_merge($formData, ['campaign' => $campaign]));
    }

    /**
     * Update existing campaign
     */
    public function update(Request $request, int $id): JsonResponse|RedirectResponse
    {
        // Check permissions
        if (! checkPermission(['campaigns.edit'])) {
            if ($request->ajax()) {
                return response()->json(['success' => false, 'message' => t('access_denied')], 403);
            }

            return redirect()->route('admin.dashboard')->with('error', t('access_denied_note'));
        }

        try {
            DB::beginTransaction();

            $campaign = Campaign::findOrFail($id);

            // Check ownership if not admin
            if (! Auth::user()->is_admin && $campaign->created_by !== auth()->id()) {
                throw new \Exception(t('access_denied'));
            }

            // Validate the request
            $validatedData = $this->validateCampaignData($request, $id);

            // Handle file upload if present
            $filename = $this->handleCampaignFileUpload($request, $campaign);

            // Update campaign
            $this->updateCampaign($campaign, $validatedData, $filename);

            // Recreate campaign details
            $campaign->details()->delete();
            $this->createCampaignDetails($campaign, $validatedData);

            DB::commit();

            if ($request->ajax()) {
                return response()->json([
                    'success'  => true,
                    'message'  => t('campaign_updated_successfully'),
                    'redirect' => route('admin.campaigns.list'),
                ]);
            }

            return redirect()->route('admin.campaigns.list')
                ->with('success', t('campaign_updated_successfully'));
        } catch (\Exception $e) {
            DB::rollBack();

            Log::error('Campaign update failed', [
                'campaign_id' => $id,
                'error'       => $e->getMessage(),
                'user_id'     => auth()->id(),
            ]);

            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => t('campaign_update_failed') . ': ' . $e->getMessage(),
                ], 500);
            }

            return back()->withInput()
                ->with('error', t('campaign_update_failed') . ': ' . $e->getMessage());
        }
    }

    /**
     * Delete campaign
     */
    public function destroy(int $id): JsonResponse|RedirectResponse
    {
        // Check permissions
        if (! checkPermission(['campaigns.delete'])) {
            if (request()->ajax()) {
                return response()->json(['success' => false, 'message' => t('access_denied')], 403);
            }

            return redirect()->route('admin.dashboard')->with('error', t('access_denied_note'));
        }
        try {
            $campaign = Campaign::findOrFail($id);

            // Check ownership if not admin
            if (! Auth::user()->is_admin && $campaign->created_by !== auth()->id()) {
                throw new \Exception(t('access_denied'));
            }

            // Delete associated file if exists
            if ($campaign->filename) {
                Storage::disk('public')->delete($campaign->filename);
            }

            // Delete campaign (cascade will handle details)
            $campaign->delete();

            if (request()->ajax()) {
                return response()->json([
                    'success' => true,
                    'message' => t('campaign_deleted_successfully'),
                ]);
            }
            session()->flash('notification', ['type' => 'success', 'message' => t('campaign_deleted_successfully')]);

            return redirect()->route('admin.campaigns.list')
                ->with('success', t('campaign_deleted_successfully'));
        } catch (\Exception $e) {
            Log::error('Campaign deletion failed', [
                'campaign_id' => $id,
                'error'       => $e->getMessage(),
                'user_id'     => auth()->id(),
            ]);

            if (request()->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => t('campaign_delete_failed') . ': ' . $e->getMessage(),
                ], 500);
            }

            return back()->with('error', t('campaign_delete_failed') . ': ' . $e->getMessage());
        }
    }

    // =================================================================
    // AJAX HELPER METHODS (Dynamic Content Loading)
    // =================================================================

    /**
     * Get contacts based on filters (AJAX)
     */
    public function getContacts(Request $request): JsonResponse
    {
        try {
            $contacts = $this->loadFilteredContacts($request->all());

            return response()->json([
                'success' => true,
                'data'    => $contacts,
                'count'   => $contacts->count(),
            ]);
        } catch (\Exception $e) {
            Log::error('Failed to load contacts', [
                'error'   => $e->getMessage(),
                'filters' => $request->all(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to load contacts: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get template data (AJAX)
     */
    public function getTemplate(Request $request): JsonResponse
    {
        try {
            $templateId = $request->input('template_id');
            $template   = WhatsappTemplate::where('template_id', $templateId)->firstOrFail();

            $templateData = $this->processTemplateData($template);

            return response()->json([
                'success' => true,
                'data'    => $templateData,
            ]);
        } catch (\Exception $e) {
            Log::error('Failed to load template', [
                'template_id' => $request->input('template_id'),
                'error'       => $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to load template: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Count contacts based on filters (AJAX)
     */
    public function countContacts(Request $request): JsonResponse
    {
        try {
            $count = $this->calculateContactCount($request->all());

            return response()->json([
                'success' => true,
                'count'   => $count,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to count contacts: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Handle file upload (AJAX)
     */
    public function uploadFile(Request $request): JsonResponse
    {
        try {
            // Validate file
            $this->validateFileUpload($request);

            // Handle upload
            $filename = $this->processFileUpload($request->file('file'), $request->input('type'));

            return response()->json([
                'success'  => true,
                'filename' => $filename,
                'url'      => Storage::disk('public')->url($filename),
                'message'  => 'File uploaded successfully',
            ]);
        } catch (\Exception $e) {
            Log::error('File upload failed', [
                'error'     => $e->getMessage(),
                'file_name' => $request->file('file')?->getClientOriginalName(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'File upload failed: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Validate form data (AJAX)
     */
    public function validateForm(Request $request): JsonResponse
    {
        try {
            $rules = $this->getValidationRules($request->input('step', 'all'));

            $validator = Validator::make($request->all(), $rules);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors'  => $validator->errors(),
                ], 422);
            }

            return response()->json([
                'success' => true,
                'message' => 'Validation passed',
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed: ' . $e->getMessage(),
            ], 500);
        }
    }

    // =================================================================
    // PRIVATE HELPER METHODS (Business Logic)
    // =================================================================

    /**
     * Check if WhatsApp is disconnected
     */
    private function isWhatsAppDisconnected(): bool
    {
        return empty(get_setting('whatsapp.is_webhook_connected')) || empty(get_setting('whatsapp.is_whatsmark_connected')) || empty(get_setting('whatsapp.wm_default_phone_number'));
    }

    /**
     * Get initial form data
     */
    private function getFormInitialData(?Campaign $campaign = null): array
    {
        // Decode existing parameters if campaign exists
        $existingVariables = [
            'header' => [],
            'body'   => [],
            'footer' => [],
        ];

        $existingFile     = null;
        $selectedContacts = [];

        if ($campaign) {
            $existingVariables['header'] = json_decode($campaign->header_params ?? '[]', true) ?: [];
            $existingVariables['body']   = json_decode($campaign->body_params ?? '[]', true) ?: [];
            $existingVariables['footer'] = json_decode($campaign->footer_params ?? '[]', true) ?: [];

            // Get existing file info
            if ($campaign->filename) {
                $existingFile = [
                    'filename' => $campaign->filename,
                    'url'      => Storage::disk('public')->url($campaign->filename),
                ];
            }

            // Get selected contacts from campaign details
            $selectedContacts = $campaign->details()->pluck('rel_id')->toArray();
        }

        return [
            'templates'         => WhatsappTemplate::where('status', 'APPROVED')->get(),
            'statuses'          => Status::select('id', 'name')->orderBy('name')->get(),
            'sources'           => Source::select('id', 'name')->orderBy('name')->get(),
            'mergeFields'       => $this->getMergeFieldsData(),
            'relationTypes'     => ['lead' => 'Lead', 'customer' => 'Customer'],
            'campaign'          => $campaign,
            'existingVariables' => $existingVariables,
            'existingFile'      => $existingFile,
            'selectedContacts'  => $selectedContacts,
        ];
    }

    /**
     * Get merge fields data
     */
    private function getMergeFieldsData(): string
    {
        $mergeFieldsService = app(MergeFields::class);

        $fields = array_merge(
            $mergeFieldsService->getFieldsForTemplate('other-group'),
            $mergeFieldsService->getFieldsForTemplate('contact-group')
        );

        return json_encode(array_map(fn ($field) => [
            'key'   => ucfirst($field['name']),
            'value' => $field['key'],
        ], $fields));
    }

    // =================================================================
    // VALIDATION METHODS
    // =================================================================

    /**
     * Validate campaign data
     */
    private function validateCampaignData(Request $request, ?int $campaignId = null): array
    {
        $rules = [
            'campaign_name' => [
                'required',
                'min:3',
                'max:255',
                'unique:campaigns,name' . ($campaignId ? ',' . $campaignId : ''),
                new PurifiedInput(t('sql_injection_error')),
            ],
            'rel_type'              => 'required|in:lead,customer',
            'template_id'           => 'required|exists:whatsapp_templates,template_id',
            'send_now'              => 'required|in:0,1,true,false', // Accept both string and boolean
            'select_all'            => 'required|in:0,1,true,false', // Accept both string and boolean
            'relation_type_dynamic' => 'array', // Remove conditional for now
            'status_name'           => 'nullable|exists:statuses,id',
            'source_name'           => 'nullable|exists:sources,id',
            'headerInputs'          => 'array',
            'headerInputs.*'        => [new PurifiedInput(t('dynamic_input_error'))],
            'bodyInputs'            => 'array',
            'bodyInputs.*'          => [new PurifiedInput(t('dynamic_input_error'))],
            'footerInputs'          => 'array',
            'footerInputs.*'        => [new PurifiedInput(t('dynamic_input_error'))],
            'file'                  => 'nullable|file',
        ];

        // Add scheduled_send_time validation only if needed
        $sendNow = $request->input('send_now');
        if (! in_array($sendNow, ['1', 1, true, 'true'], true)) {
            $rules['scheduled_send_time'] = 'required|date|after:now';
        }

        // Add contact validation only if not select all
        $selectAll = $request->input('select_all');
        if (! in_array($selectAll, ['1', 1, true, 'true'], true)) {
            $rules['relation_type_dynamic'] = 'required|array|min:1';
        }

        $validatedData = $request->validate($rules);

        // Normalize boolean values
        $validatedData['send_now']   = in_array($validatedData['send_now'], ['1', 1, true, 'true'], true);
        $validatedData['select_all'] = in_array($validatedData['select_all'], ['1', 1, true, 'true'], true);

        return $validatedData;
    }

    /**
     * Get validation rules for specific steps
     */
    private function getValidationRules(string $step): array
    {
        $allRules = [
            'campaign_name' => 'required|min:3|max:255',
            'rel_type'      => 'required|in:lead,customer',
            'template_id'   => 'required',
            // Add more rules as needed
        ];

        return match ($step) {
            'basic'    => array_intersect_key($allRules, array_flip(['campaign_name', 'rel_type'])),
            'template' => array_intersect_key($allRules, array_flip(['template_id'])),
            'all'      => $allRules,
            default    => $allRules
        };
    }

    // =================================================================
    // BUSINESS LOGIC METHODS (Add these to your existing controller)
    // =================================================================

    /**
     * Load filtered contacts based on request parameters
     */
    private function loadFilteredContacts(array $filters): Collection
    {
        $relType   = $filters['rel_type']   ?? null;
        $statusId  = $filters['status_id']  ?? null;
        $sourceId  = $filters['source_id']  ?? null;
        $selectAll = $filters['select_all'] ?? false;

        if (! $relType) {
            return collect([]);
        }

        $query = Contact::where('type', $relType)
            ->where('is_enabled', 1);

        // Apply permission-based filtering
        if (! Auth::user()->is_admin && checkPermission('contact.view_own')) {
            $query->where('assigned_id', auth()->id());
        }

        // Apply status filter
        if ($statusId) {
            $query->where('status_id', $statusId);
        }

        // Apply source filter
        if ($sourceId) {
            $query->where('source_id', $sourceId);
        }

        return $query->select('id', 'firstname', 'lastname', 'email', 'phone')
            ->orderBy('firstname')
            ->get();
    }

    /**
     * Calculate contact count based on filters
     */
    private function calculateContactCount(array $filters): int
    {
        $relType          = $filters['rel_type']          ?? null;
        $statusId         = $filters['status_id']         ?? null;
        $sourceId         = $filters['source_id']         ?? null;
        $selectAll        = $filters['select_all']        ?? false;
        $selectedContacts = $filters['selected_contacts'] ?? [];

        if (! $relType) {
            return 0;
        }

        // If specific contacts are selected, return their count
        if (! $selectAll && ! empty($selectedContacts)) {
            return count($selectedContacts);
        }

        // If select all is enabled, count all matching contacts
        $query = Contact::where('type', $relType)
            ->where('is_enabled', 1);

        // Apply permission-based filtering
        if (! Auth::user()->is_admin && checkPermission('contact.view_own')) {
            $query->where('assigned_id', auth()->id());
        }

        // Apply filters
        if ($statusId) {
            $query->where('status_id', $statusId);
        }

        if ($sourceId) {
            $query->where('source_id', $sourceId);
        }

        return $query->count();
    }

    /**
     * Process template data for frontend
     */
    private function processTemplateData(WhatsappTemplate $template): array
    {
        return [
            'id'       => $template->template_id,
            'name'     => $template->template_name,
            'language' => $template->language,
            'header'   => [
                'format'       => $template->header_data_format  ?? 'TEXT',
                'text'         => $template->header_data_text    ?? '',
                'params_count' => $template->header_params_count ?? 0,
            ],
            'body' => [
                'text'         => $template->body_data         ?? '',
                'params_count' => $template->body_params_count ?? 0,
            ],
            'footer' => [
                'text'         => $template->footer_data         ?? '',
                'params_count' => $template->footer_params_count ?? 0,
            ],
            'buttons'            => $this->parseTemplateButtons($template->buttons_data),
            'allowed_file_types' => $this->getAllowedFileTypes($template->header_data_format),
            'max_file_size'      => $this->getMaxFileSize($template->header_data_format),
        ];
    }

    /**
     * Parse template buttons data
     */
    private function parseTemplateButtons(?string $buttonsData): array
    {
        if (empty($buttonsData)) {
            return [];
        }

        try {
            $buttons = json_decode($buttonsData, true);

            return is_array($buttons) ? $buttons : [];
        } catch (\Exception $e) {
            Log::warning('Failed to parse template buttons', [
                'buttons_data' => $buttonsData,
                'error'        => $e->getMessage(),
            ]);

            return [];
        }
    }

    /**
     * Get allowed file types based on template format
     */
    private function getAllowedFileTypes(?string $format): array
    {
        $extensions = get_meta_allowed_extension();

        return match ($format) {
            'IMAGE' => [
                'extensions' => $extensions['image']['extension'] ?? '.jpeg,.png',
                'accept'     => 'image/*',
            ],
            'VIDEO' => [
                'extensions' => $extensions['video']['extension'] ?? '.mp4,.3gp',
                'accept'     => 'video/*',
            ],
            'DOCUMENT' => [
                'extensions' => $extensions['document']['extension'] ?? '.pdf,.doc,.docx',
                'accept'     => '.pdf,.doc,.docx,.txt,.xls,.xlsx,.ppt,.pptx',
            ],
            'AUDIO' => [
                'extensions' => $extensions['audio']['extension'] ?? '.mp3,.aac',
                'accept'     => 'audio/*',
            ],
            default => [
                'extensions' => '',
                'accept'     => '',
            ]
        };
    }

    /**
     * Get max file size based on template format
     */
    private function getMaxFileSize(?string $format): int
    {
        $extensions = get_meta_allowed_extension();

        return match ($format) {
            'IMAGE'    => ($extensions['image']['size'] ?? 5)      * 1024 * 1024, // Convert MB to bytes
            'VIDEO'    => ($extensions['video']['size'] ?? 16)     * 1024 * 1024,
            'DOCUMENT' => ($extensions['document']['size'] ?? 100) * 1024 * 1024,
            'AUDIO'    => ($extensions['audio']['size'] ?? 16)     * 1024 * 1024,
            default    => 5                                        * 1024 * 1024 // 5MB default
        };
    }

    // =================================================================
    // CAMPAIGN MANAGEMENT METHODS
    // =================================================================

    /**
     * Create new campaign
     */
    private function createCampaign(array $validatedData, ?string $filename): Campaign
    {
        $scheduledTime = $this->processScheduledTime($validatedData);

        return Campaign::create([
            'name'                => $validatedData['campaign_name'],
            'rel_type'            => $validatedData['rel_type'],
            'template_id'         => $validatedData['template_id'],
            'scheduled_send_time' => $scheduledTime,
            'send_now'            => $validatedData['send_now']   ?? false,
            'select_all'          => $validatedData['select_all'] ?? false,
            'sending_count'       => $this->calculateSendingCount($validatedData),
            'header_params'       => $this->encodeParams($validatedData['headerInputs'] ?? []),
            'body_params'         => $this->encodeParams($validatedData['bodyInputs'] ?? []),
            'footer_params'       => $this->encodeParams($validatedData['footerInputs'] ?? []),
            'filename'            => $filename,
            'rel_data'            => $this->encodeRelationData($validatedData),
            'status'              => 'draft',
            'created_by'          => auth()->id(),
            'created_at'          => now(),
            'updated_at'          => now(),
        ]);
    }

    /**
     * Update existing campaign
     */
    private function updateCampaign(Campaign $campaign, array $validatedData, ?string $filename): void
    {
        $scheduledTime = $this->processScheduledTime($validatedData);
        $campaign->update([
            'name'                => $validatedData['campaign_name'],
            'rel_type'            => $validatedData['rel_type'],
            'template_id'         => $validatedData['template_id'],
            'scheduled_send_time' => $scheduledTime,
            'send_now'            => $validatedData['send_now']   ?? false,
            'select_all'          => $validatedData['select_all'] ?? false,
            'sending_count'       => $this->calculateSendingCount($validatedData),
            'header_params'       => $this->encodeParams($validatedData['headerInputs'] ?? []),
            'body_params'         => $this->encodeParams($validatedData['bodyInputs'] ?? []),
            'footer_params'       => $this->encodeParams($validatedData['footerInputs'] ?? []),
            'filename'            => $filename ?? $campaign->filename,
            'rel_data'            => $this->encodeRelationData($validatedData),
            'updated_at'          => now(),
        ]);
    }

    /**
     * Process scheduled time
     */
    private function processScheduledTime(array $validatedData): ?string
    {
        if ($validatedData['send_now'] ?? false) {

            return now()->toDateTimeString();
        }

        if (empty($validatedData['scheduled_send_time'])) {
            return null;
        }

        try {
            // Convert from user format to database format
            $dateFormat = get_setting('general.date_format', 'd-m-Y');
            $timeFormat = get_setting('general.time_format') == '12' ? 'h:i A' : 'H:i';
            $format     = $dateFormat . ' ' . $timeFormat;

            return Carbon::createFromFormat($format, $validatedData['scheduled_send_time'])
                ->toDateTimeString();
        } catch (\Exception $e) {
            throw new \Exception('Invalid date/time format: ' . $e->getMessage());
        }
    }

    /**
     * Calculate sending count
     */
    private function calculateSendingCount(array $validatedData): int
    {
        if ($validatedData['select_all'] ?? false) {
            return $this->calculateContactCount($validatedData);
        }

        return count($validatedData['relation_type_dynamic'] ?? []);
    }

    /**
     * Encode parameters array to JSON
     */
    private function encodeParams(array $params): string
    {
        return json_encode(array_values(array_filter($params)));
    }

    /**
     * Encode relation data
     */
    private function encodeRelationData(array $validatedData): string
    {
        return json_encode([
            'status_id' => $validatedData['status_name'] ?? null,
            'source_id' => $validatedData['source_name'] ?? null,
        ]);
    }

    // =================================================================
    // CAMPAIGN DETAILS CREATION
    // =================================================================

    /**
     * Create campaign details for selected contacts
     */
    private function createCampaignDetails(Campaign $campaign, array $validatedData): void
    {
        $template = WhatsappTemplate::where('template_id', $campaign->template_id)->firstOrFail();
        $contacts = $this->getSelectedContacts($validatedData);

        if ($contacts->isEmpty()) {
            throw new \Exception('No contacts selected for campaign');
        }

        $campaignDetails = [];
        $headerInputs    = $validatedData['headerInputs'] ?? [];
        $bodyInputs      = $validatedData['bodyInputs']   ?? [];
        $footerInputs    = $validatedData['footerInputs'] ?? [];

        foreach ($contacts as $contact) {
            $campaignDetails[] = [
                'campaign_id'      => $campaign->id,
                'rel_id'           => $contact->id,
                'rel_type'         => $campaign->rel_type,
                'header_message'   => $this->parseMessageVariables($template->header_data_text, $headerInputs, $contact),
                'body_message'     => $this->parseMessageVariables($template->body_data, $bodyInputs, $contact),
                'footer_message'   => $this->parseMessageVariables($template->footer_data, $footerInputs, $contact),
                'status'           => 1,
                'message_status'   => 'pending',
                'whatsapp_id'      => null,
                'response_message' => '',
                'created_at'       => now(),
                'updated_at'       => now(),
            ];
        }

        // Bulk insert for better performance
        CampaignDetail::insert($campaignDetails);
    }

    /**
     * Get selected contacts for campaign
     */
    private function getSelectedContacts(array $validatedData): Collection
    {
        if ($validatedData['select_all'] ?? false) {
            return $this->loadFilteredContacts($validatedData);
        }

        $contactIds = $validatedData['relation_type_dynamic'] ?? [];

        if (empty($contactIds)) {
            return collect([]);
        }

        return Contact::whereIn('id', $contactIds)
            ->where('type', $validatedData['rel_type'])
            ->where('is_enabled', 1)
            ->get();
    }

    /**
     * Parse message variables with contact data
     */
    private function parseMessageVariables(?string $message, array $params, $contact): ?string
    {
        if (empty($message)) {
            return null;
        }

        $parsedMessage = $message;

        // Replace numbered parameters {{1}}, {{2}}, etc.
        foreach ($params as $index => $param) {
            $placeholder   = '{{' . ($index + 1) . '}}';
            $value         = $param ?? '';
            $parsedMessage = str_replace($placeholder, $value, $parsedMessage);
        }

        // Replace contact merge fields
        $parsedMessage = $this->replaceMergeFields($parsedMessage, $contact);

        return $parsedMessage;
    }

    /**
     * Replace merge fields with contact data
     */
    private function replaceMergeFields(string $message, $contact): string
    {
        $mergeFields = [
            '{{contact_first_name}}' => $contact->firstname ?? '',
            '{{contact_last_name}}'  => $contact->lastname  ?? '',
            '{{contact_full_name}}'  => trim(($contact->firstname ?? '') . ' ' . ($contact->lastname ?? '')),
            '{{contact_email}}'      => $contact->email   ?? '',
            '{{contact_phone}}'      => $contact->phone   ?? '',
            '{{contact_company}}'    => $contact->company ?? '',
            '{{business_name}}'      => get_setting('general.site_name', 'Business'),
            '{{current_date}}'       => now()->format('Y-m-d'),
            '{{current_time}}'       => now()->format('H:i:s'),
        ];

        return str_replace(array_keys($mergeFields), array_values($mergeFields), $message);
    }

    // =================================================================
    // FILE HANDLING METHODS
    // =================================================================

    /**
     * Handle campaign file upload
     */
    private function handleCampaignFileUpload(Request $request, ?Campaign $existingCampaign = null): ?string
    {

        if (! $request->hasFile('file')) {
            return $existingCampaign?->filename;
        }

        $file       = $request->file('file');
        $templateId = $request->input('template_id');

        // Get template to determine file type requirements
        $template = WhatsappTemplate::where('template_id', $templateId)->first();
        if (! $template) {
            throw new \Exception('Template not found');
        }

        // Validate file
        $this->validateFileUpload($request, $template->header_data_format);

        // Delete existing file if updating
        if ($existingCampaign && $existingCampaign->filename) {
            Storage::disk('public')->delete($existingCampaign->filename);
        }

        // Process upload
        return $this->processFileUpload($file, $template->header_data_format);
    }

    /**
     * Validate file upload
     */
    private function validateFileUpload(Request $request, ?string $expectedFormat = null): void
    {
        $file = $request->file('file');

        if (! $file || ! $file->isValid()) {
            throw new \Exception('Invalid file upload');
        }

        // Get file extension and MIME type
        $extension = strtolower($file->getClientOriginalExtension());
        $mimeType  = $file->getMimeType();
        $fileSize  = $file->getSize();

        // Get allowed file types and sizes
        $extensions = get_meta_allowed_extension();

        // Validate based on expected format
        if ($expectedFormat) {
            $formatKey = strtolower($expectedFormat);

            if (! isset($extensions[$formatKey])) {
                throw new \Exception('Unsupported file format: ' . $expectedFormat);
            }

            $allowedExtensions = explode(',', $extensions[$formatKey]['extension']);
            $allowedExtensions = array_map('trim', $allowedExtensions);
            $maxSize           = $extensions[$formatKey]['size'] * 1024 * 1024; // Convert MB to bytes

            // Check extension
            if (! in_array('.' . $extension, $allowedExtensions)) {
                throw new \Exception('Invalid file extension. Allowed: ' . implode(', ', $allowedExtensions));
            }

            // Check file size
            if ($fileSize > $maxSize) {
                throw new \Exception('File size too large. Maximum: ' . $extensions[$formatKey]['size'] . 'MB');
            }

            // Validate MIME type for security
            $this->validateMimeType($mimeType, $formatKey);
        }
    }

    /**
     * Validate MIME type for security
     */
    private function validateMimeType(string $mimeType, string $format): void
    {
        $allowedMimeTypes = [
            'image'    => ['image/jpeg', 'image/png', 'image/gif', 'image/webp'],
            'video'    => ['video/mp4', 'video/3gpp', 'video/quicktime'],
            'document' => [
                'application/pdf',
                'application/msword',
                'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
                'application/vnd.ms-excel',
                'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
                'application/vnd.ms-powerpoint',
                'application/vnd.openxmlformats-officedocument.presentationml.presentation',
                'text/plain',
            ],
            'audio' => ['audio/mpeg', 'audio/mp4', 'audio/ogg', 'audio/wav', 'audio/aac'],
        ];

        if (isset($allowedMimeTypes[$format]) && ! in_array($mimeType, $allowedMimeTypes[$format])) {
            throw new \Exception('Invalid file type. Detected: ' . $mimeType);
        }
    }

    /**
     * Process file upload and return filename
     */
    private function processFileUpload(UploadedFile $file, string $format): string
    {
        // Generate secure filename
        $originalName = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
        $extension    = $file->getClientOriginalExtension();
        $filename     = Str::slug($originalName) . '_' . time() . '.' . $extension;

        // Determine storage directory
        $directory = match (strtolower($format)) {
            'image'    => 'campaigns/images',
            'video'    => 'campaigns/videos',
            'document' => 'campaigns/documents',
            'audio'    => 'campaigns/audio',
            default    => 'campaigns'
        };

        // Store file
        $path = $file->storeAs($directory, $filename, 'public');

        if (! $path) {
            throw new \Exception('Failed to store file');
        }

        return $path;
    }

    // =================================================================
    // UTILITY METHODS
    // =================================================================

    /**
     * Format JSON response
     */
    private function formatResponse(bool $success, $data = null, string $message = '', array $errors = []): array
    {
        return [
            'success'   => $success,
            'data'      => $data,
            'message'   => $message,
            'errors'    => $errors,
            'timestamp' => now()->toISOString(),
        ];
    }

    /**
     * Log campaign activity
     */
    private function logCampaignActivity(string $action, array $data = [], ?Campaign $campaign = null): void
    {
        Log::info('Campaign activity: ' . $action, [
            'campaign_id' => $campaign?->id,
            'user_id'     => auth()->id(),
            'action'      => $action,
            'data'        => $data,
            'timestamp'   => now(),
        ]);
    }

    /**
     * Get campaign statistics
     */
    private function getCampaignStats(Campaign $campaign): array
    {
        $details = $campaign->details();

        return [
            'total_contacts'  => $details->count(),
            'sent_count'      => $details->where('status', 1)->count(),
            'pending_count'   => $details->where('status', 0)->count(),
            'failed_count'    => $details->where('status', -1)->count(),
            'delivered_count' => $details->where('message_status', 'delivered')->count(),
            'read_count'      => $details->where('message_status', 'read')->count(),
        ];
    }
}
