import fs from 'fs/promises';
import path from 'path';
import { pathToFileURL } from 'url';

async function collectModuleAssetsPaths(paths, modulesPath) {
  modulesPath = path.join(__dirname, modulesPath);
  const moduleStatusesPath = path.join(__dirname, 'modules_statuses.json');

  try {
    // Read module_statuses.json
    const moduleStatusesContent = await fs.readFile(moduleStatusesPath, 'utf-8');
    const moduleStatuses = JSON.parse(moduleStatusesContent);

    // Read module directories
    const moduleDirectories = await fs.readdir(modulesPath);

    for (const moduleDir of moduleDirectories) {
      if (moduleDir === '.DS_Store') {
        // Skip .DS_Store directory
        continue;
      }

      // Check if the module is enabled (status is true)
      if (moduleStatuses[moduleDir] === true) {
        const viteConfigPath = path.join(modulesPath, moduleDir, 'vite.config.js');

        try {
          await fs.access(viteConfigPath);
          // Convert to a file URL for Windows compatibility
          const moduleConfigURL = pathToFileURL(viteConfigPath);

          // Import the module-specific Vite configuration
          const moduleConfig = await import(moduleConfigURL.href);

          if (moduleConfig.paths && Array.isArray(moduleConfig.paths)) {
            paths.push(...moduleConfig.paths);
          }
        } catch (error) {
          // vite.config.js does not exist, skip this module
        }
      }
    }
  } catch (error) {
    console.error(`Error reading module statuses or module configurations: ${error}`);
  }

  return paths;
}

/**
 * Collect all enabled module view paths for watching
 * @param {string} modulesPath - Path to modules directory
 * @returns {Promise<string[]>} Array of view paths to watch
 */
async function collectModuleViewPaths(modulesPath = 'Modules') {
  const viewPaths = [];
  const fullModulesPath = path.join(__dirname, modulesPath);
  const moduleStatusesPath = path.join(__dirname, 'modules_statuses.json');

  try {
    // Read module_statuses.json
    const moduleStatusesContent = await fs.readFile(moduleStatusesPath, 'utf-8');
    const moduleStatuses = JSON.parse(moduleStatusesContent);

    // Read module directories
    const moduleDirectories = await fs.readdir(fullModulesPath);

    for (const moduleDir of moduleDirectories) {
      if (moduleDir === '.DS_Store') {
        continue;
      }

      // Check if the module is enabled
      if (moduleStatuses[moduleDir] === true) {
        const moduleViewsPath = path.join(modulesPath, moduleDir, 'Resources/views');
        const fullViewsPath = path.join(__dirname, moduleViewsPath);

        try {
          // Check if views directory exists
          await fs.access(fullViewsPath);
          // Add all blade files in this module's views directory
          viewPaths.push(`${moduleViewsPath}/**/*.blade.php`);

          console.log(`✓ Watching views for module: ${moduleDir}`);
        } catch (error) {
          // Views directory doesn't exist, skip
          console.log(`- No views directory for module: ${moduleDir}`);
        }
      }
    }
  } catch (error) {
    console.error(`Error collecting module view paths: ${error}`);
  }

  return viewPaths;
}

/**
 * Collect all enabled module asset and Livewire component paths
 * @param {string} modulesPath - Path to modules directory
 * @returns {Promise<string[]>} Array of paths to watch
 */
async function collectModuleWatchPaths(modulesPath = 'Modules') {
  const watchPaths = [];
  const fullModulesPath = path.join(__dirname, modulesPath);
  const moduleStatusesPath = path.join(__dirname, 'modules_statuses.json');

  try {
    const moduleStatusesContent = await fs.readFile(moduleStatusesPath, 'utf-8');
    const moduleStatuses = JSON.parse(moduleStatusesContent);
    const moduleDirectories = await fs.readdir(fullModulesPath);

    for (const moduleDir of moduleDirectories) {
      if (moduleDir === '.DS_Store') continue;

      if (moduleStatuses[moduleDir] === true) {
        const moduleBasePath = `${modulesPath}/${moduleDir}`;

        // Watch common directories that might affect the frontend
        const pathsToWatch = [
          `${moduleBasePath}/Resources/views/**/*.blade.php`,     // Blade templates
          `${moduleBasePath}/Resources/assets/**/*`,              // Assets
          `${moduleBasePath}/Http/Livewire/**/*.php`,            // Livewire components (old location)
          `${moduleBasePath}/Livewire/**/*.php`,                 // Livewire components (new location)
          `${moduleBasePath}/Providers/**/*.php`,                // Service providers
        ];

        // Check which paths actually exist
        for (const watchPath of pathsToWatch) {
          const basePath = watchPath.split('/**')[0];
          const fullPath = path.join(__dirname, basePath);

          try {
            await fs.access(fullPath);
            watchPaths.push(watchPath);
          } catch (error) {
            // Path doesn't exist, skip
          }
        }
      }
    }
  } catch (error) {
    console.error(`Error collecting module watch paths: ${error}`);
  }

  return watchPaths;
}

export default collectModuleAssetsPaths;
export { collectModuleViewPaths, collectModuleWatchPaths };