import { defineConfig } from "vite";
import laravel from "laravel-vite-plugin";
import path from "path";
import fs from "fs";
import vue from '@vitejs/plugin-vue';
import collectModuleAssetsPaths, {
    collectModuleViewPaths,
    collectModuleWatchPaths
} from "./vite-module-loader.js";

const themeFilePath = path.resolve(__dirname, "theme.json");
const activeTheme = fs.existsSync(themeFilePath)
    ? JSON.parse(fs.readFileSync(themeFilePath, "utf8")).name
    : "anchor";
console.log(`Active theme: ${activeTheme}`);

export default defineConfig(async () => {
    // Collect module asset paths
    let moduleAssets = [];
    try {
        moduleAssets = await collectModuleAssetsPaths([], "Modules");
        console.log(`Loaded ${moduleAssets.length} module assets`);
    } catch (error) {
        console.warn("Failed to load module assets:", error.message);
    }

    // Collect module view paths for watching
    let moduleViewPaths = [];
    try {
        moduleViewPaths = await collectModuleViewPaths("Modules");
        console.log(`Watching ${moduleViewPaths.length} module view directories`);
    } catch (error) {
        console.warn("Failed to load module view paths:", error.message);
    }

    // Collect all module watch paths (comprehensive)
    let moduleWatchPaths = [];
    try {
        moduleWatchPaths = await collectModuleWatchPaths("Modules");
        console.log(`Total module watch paths: ${moduleWatchPaths.length}`);
    } catch (error) {
        console.warn("Failed to load module watch paths:", error.message);
    }

    const inputFiles = [
        "resources/css/app.css",
        "resources/js/app.js",
        ...moduleAssets, // Add module assets
    ];

    // Combine all watch paths
    const allWatchPaths = [
        `resources/themes/${activeTheme}/**/*`,
        "resources/views/**/*.blade.php",           // Core views
        "app/Http/Livewire/**/*.php",              // Core Livewire components
        "app/Livewire/**/*.php",                   // Core Livewire components (new location)
        ...moduleViewPaths,                        // Module views (simplified)
        ...moduleWatchPaths,                       // All module watch paths (comprehensive)
    ];

    console.log("📁 Watch paths configured:");
    allWatchPaths.forEach(path => console.log(`   • ${path}`));

    return {
        plugins: [
            laravel({
                input: inputFiles,
                refresh: allWatchPaths,  // Enhanced refresh paths
            }),
            vue({
                template: {
                    transformAssetUrls: {
                        base: null,
                        includeAbsolute: false,
                    },
                },
            }),
        ],
        server: {
            cors: true,
            host: '0.0.0.0',  // Allow external connections
            port: 5173,       // Default Vite port
            strictPort: false, // Allow port to be changed if occupied
        },
        resolve: {
            alias: {
                '@': '/resources/js',
                vue: 'vue/dist/vue.esm-bundler.js',
            },
        },
        build: {
            chunkSizeWarningLimit: 1000, // Increased from default 500kb to 1000kb
            sourcemap: false,            // Disable sourcemaps in production for faster builds
            rollupOptions: {
                output: {
                    manualChunks: {
                    },
                },
            },
        },
        // Enhanced file watching
        optimizeDeps: {
            include: [
                '@nextapps-be/livewire-sortablejs',
                'tippy.js',
                'tom-select',
            ],
        },
    };
});